CREATE OR REPLACE FUNCTION /*tabk.*/tableident_allfields(
  IN  _abk abk,
  OUT ident             character varying,
  OUT aknr              character varying,
  OUT annr              character varying,
  OUT keyvalue          character varying,
  OUT tablename         character varying,
  OUT link_ident_short  character varying,
  OUT txt               text
  )
  RETURNS record
  AS $$
  DECLARE _tname varchar;
          _tdbrid varchar;
  BEGIN

      IF _abk.ab_tablename IS null THEN
        -- nur im FALL direkter PA hat tablename keine Wert
        _tname := 'ldsdok';
        _tdbrid := dbrid FROM ldsdok WHERE ld_abk = _abk.ab_ix;
      ELSE
        _tname := _abk.ab_tablename;
        _tdbrid := _abk.ab_dbrid;
      END IF;

      SELECT f.ident,
             f.aknr,
             f.annr,
             f.keyvalue,
             f.txt
        INTO ident,
             aknr,
             annr,
             keyvalue,
             txt
        FROM tableident_allfields(_tname, _tdbrid) f;

      tablename := _tname;

      link_ident_short := tableident_tablename_short(_tname, _tdbrid);

      RETURN;

  END $$ LANGUAGE plpgsql;

CREATE OR REPLACE FUNCTION tabk.abk_with_children_get(start_ids integer[], CheckUABK boolean)
    RETURNS TABLE(ab_ix integer, level integer)
    AS $$
    DECLARE
        returned integer[];           -- Bereits zurückgegebene Knoten
        queue integer[];              -- Warteschlange für die Verarbeitung
        level_queue integer[];        -- Gleiche Länge wie 'start_ids'
        current_ab_ix integer;        -- Aktuell verarbeiteter Knoten
        current_level integer;        -- Level des aktuellen Knotens
        child_ab_ix integer;          -- Kind-Knoten aus 'tplanterm.get_all_child_abk'
    BEGIN
        IF start_ids IS null THEN
            RETURN;
        END IF;

        returned := ARRAY[]::integer[];                                     -- Bereits zurückgegebene Knoten
        queue := start_ids;                                                 -- Warteschlange für die Verarbeitung
        level_queue := array_fill( 0, ARRAY[array_length(start_ids, 1)] );  -- Gleiche Länge wie 'start_ids'


        -- Solange noch Knoten in der Warteschlange sind
        WHILE array_length(queue, 1) > 0 LOOP
            -- Sicherstellen, dass 'level_queue' synchron ist
            IF array_length(queue, 1) <> array_length(level_queue, 1) THEN
                RAISE EXCEPTION 'Fehler: queue und level_queue sind inkonsistent!';
            END IF;

            -- Nehme den ersten Knoten und sein Level aus der Warteschlange
            current_ab_ix := queue[1];
            current_level := level_queue[1];

            -- Entferne den ersten Knoten aus der Warteschlange
            queue := queue[2 : array_length(queue, 1)];
            level_queue := level_queue[2 : array_length(level_queue, 1)];

            -- Falls der Knoten bereits verarbeitet wurde, überspringen
            IF NOT current_ab_ix = ANY(returned) THEN
                -- Füge den Knoten zur Ergebnismenge hinzu
                returned := array_append(returned, current_ab_ix);
                ab_ix := current_ab_ix;
                level := current_level;
                RETURN NEXT;

                -- Falls CheckUABK = TRUE, Kind-Knoten abrufen
                IF CheckUABK IS true THEN
                    FOR child_ab_ix IN SELECT * FROM tplanterm.get_all_child_abk(current_ab_ix/*, false, false, false*/)
                    LOOP
                        -- Falls der Kind-Knoten noch nicht verarbeitet wurde, zur Warteschlange hinzufügen
                        IF NOT child_ab_ix = ANY(returned) THEN
                            queue := array_append(queue, child_ab_ix);
                            level_queue := array_append(level_queue, current_level + 1);
                        END IF;
                    END LOOP;
                END IF;

            END IF;
        END LOOP;

    END $$ LANGUAGE plpgsql; -- STRICT; NEIN; da CheckUABK null sein könnte und dann false interpretiert wird (Oberfläche / Vorsatzformular hat Parameter nicht)
--



-- Vergleich [AG-Nummern(wenn eingegeben ist)/AG-Reihenfolge, Kostenstellen, Zeiten (a2_tr, a2_th), AG-Text]
CREATE OR REPLACE FUNCTION tabk.abk_planauftg__compare(abix INTEGER, agn INTEGER DEFAULT NULL) RETURNS VARCHAR AS $$
  DECLARE md5_abk_green     VARCHAR;
          md5_abk_yellow    VARCHAR;
          md5_pabk_green    VARCHAR;
          md5_pabk_yellow   VARCHAR;
          planABK           INTEGER;
  BEGIN
    planABK= tabk.abk_planauftg__getPlanABK(abix);
    IF (SELECT count(*) FROM abk WHERE ab_ix IN (abix, planABK)) <> 2 THEN
        RETURN NULL;
    END IF;

    -- green:  AG-Nr, Kostenstelle, Rüstzeit, Hauptzeit, Nebenzeit, Texte
    -- yellow: AG-Nr, Kostenstelle
    SELECT md5(array_agg(array[a2_n::VARCHAR, a2_ks, a2_tr_sek::VARCHAR, a2_th_sek::VARCHAR, a2_tn_sek::VARCHAR, a2_txt, a2_txt_intern]::TEXT ORDER BY a2_n, a2_id)::TEXT),
           md5(array_agg(array[a2_n::VARCHAR, a2_ks]::TEXT ORDER BY a2_n, a2_id)::TEXT)
    FROM ab2 WHERE (agn IS NULL AND a2_ab_ix = abix) OR (agn IS NOT NULL AND a2_ab_ix = abix AND a2_n = agn)
    INTO md5_abk_green, md5_abk_yellow;

    SELECT md5(array_agg(array[a2_n::VARCHAR, a2_ks, a2_tr_sek::VARCHAR, a2_th_sek::VARCHAR, a2_tn_sek::VARCHAR, a2_txt, a2_txt_intern]::TEXT ORDER BY a2_n, a2_id)::TEXT),
           md5(array_agg(array[a2_n::VARCHAR, a2_ks]::TEXT ORDER BY a2_n, a2_id)::TEXT)
    FROM ab2 WHERE (agn IS NULL AND a2_ab_ix = planABK) OR (agn IS NOT NULL AND a2_ab_ix = planABK AND a2_n = agn)
    INTO md5_pabk_green, md5_pabk_yellow;

    IF md5_abk_green = md5_pabk_green THEN
        RETURN 'green';  -- 100% Identisch (Grün)
    ELSEIF md5_abk_yellow = md5_pabk_yellow THEN
        RETURN 'yellow'; -- Arbeitsgänge/Kostenstellen gleich (Verlauf), Zeiten/Texte anders (Gelb)
    ELSE
        RETURN 'red';    -- Große Unterschiede (Arbeitsverlauf) (Rot)
    END IF;
  END $$ LANGUAGE plpgsql;
--

--Übernahme der Änderung von Arbeitsgängen von Erstellter ABK in PlanABK
-- http://redmine.prodat-sql.de/issues/7594
CREATE OR REPLACE FUNCTION tabk.abk_planauftg__assign(abix INTEGER) RETURNS BOOLEAN AS $$
 DECLARE
    planABK INTEGER;
    status VARCHAR;
 BEGIN
    planABK = tabk.abk_planauftg__getPlanABK(abix);
    status = tabk.abk_planauftg__compare(abix);

    PERFORM TSystem.execution_flag__aquire( _flagname => 'inTerminierung' ); -- ohne dies würde evtl ab2__a_iu__termination_manual aufgerufen!

    -- Derzeit nur für den Status Grün und Gelb
    IF (status IN ('green', 'yellow')) AND (planABK IS NOT NULL) THEN
      --PlanABK soll in AG erhalten bleiben, damit Einstellungen, Prioritäten usw erhalten bleiben


      --Richtige ABK aktualisiert Arbeitsgänge der PlanABK
      IF status = 'yellow' THEN --beim "yellow" Arbeitsgänge/Kostenstellen sind schon gleich (Verlauf)
        UPDATE ab2 SET
          --Terminierung
          a2_dlz        = ab2_fert.a2_dlz,
          a2_lgz        = ab2_fert.a2_lgz,
          --Rüsten
          a2_tr         = ab2_fert.a2_tr,
          a2_zeinh_tr   = ab2_fert.a2_zeinh_tr,
          --Fertigen
          a2_th         = ab2_fert.a2_th,
          a2_zeinh_tx   = ab2_fert.a2_zeinh_tx,
          a2_tv         = ab2_fert.a2_tv,
          a2_tn         = ab2_fert.a2_tn, --Nebenzeit
          a2_tm         = ab2_fert.a2_tm,
          --Texte
          a2_txt        = ab2_fert.a2_txt,
          a2_txt_rtf    = ab2_fert.a2_txt_rtf,
          a2_txt_intern = ab2_fert.a2_txt_intern
        FROM
          ab2 AS ab2_fert
        WHERE
          ab2.a2_ab_ix = planABK        -- Ziel-ABK
          AND ab2_fert.a2_ab_ix = abix  -- Quell-ABK
          AND ab2.a2_n = ab2_fert.a2_n; -- wichtig, damit Bezug hergestellt wird, welcher AG welchen AG updated
      END IF; --bei "green" ist schon 100% Identisch

      UPDATE bdea SET ba_ix          = planABK WHERE ba_ix = abix; --Stempelungen zuerst an PlanABK hängen, AG aktueller ABK werden jetzt gelöscht
      --Richtige ABK AG Entfernen
      DELETE FROM ab2 WHERE a2_ab_ix = abix;


      --PlanABK.AG an richtige ABK umhängen (diese wurden ja bereits aktualisiert)
      UPDATE ab2    SET a2_ab_ix      = abix WHERE a2_ab_ix = planABK;
      UPDATE abk    SET ab_inplantaf  = TRUE WHERE ab_ix = abix;
      UPDATE bdea   SET ba_ix         = abix WHERE ba_ix = planABK; --Stempelungen zurückholen, PlanABK wird jetzt gleich gelöscht
      UPDATE ab2ba  SET a2ba_ab_ix    = abix WHERE a2ba_ab_ix = planABK;
      UPDATE ab2_wkstplan SET a2w_stukorr = NULL WHERE a2w_a2_id IN (SELECT a2_id FROM ab2 WHERE a2_ab_ix = abix); --Stundenkorrekturen entfernen um geänderte Plandaten auch in der PT zu haben

      PERFORM scheduling.abk__at_et__from__ab2__sync( abix );

      --PlanABK > nur noch Kopf übrig, entfernen
      DELETE FROM abk WHERE ab_ix = planABK;

      PERFORM TSystem.execution_flag__release( _flagname => 'inTerminierung' );

      RETURN TRUE;
    END IF;
    RETURN FALSE;
 END $$ LANGUAGE plpgsql;
--

-- Hole die PlanABK zum Kundenauftrag über die ausgelöste ABK der internen Bestellung
CREATE OR REPLACE FUNCTION tabk.abk_planauftg__getPlanABK(abix integer) RETURNS integer AS $$
  DECLARE agid     varchar;
          aknr     varchar;
          ldaknr   varchar;
          result   integer;
          r        record;
  BEGIN
    result:= NULL;

    -- 1. direkt scharf verbunden > manuell (zB weil indexänderung oder anderes)
    SELECT ld_planabk, ld_aknr INTO result, ldaknr FROM ldsdok WHERE ld_abk = abix;
    IF result IS NOT NULL THEN
        RETURN result;
    END IF;

    -- 2. über Kundenauftrag prüfen > funktioniert für Endprodukte
    SELECT ag_id, ag_aknr INTO agid, aknr FROM auftg WHERE ag_id = tplanterm.abk_main_auftg_id(abix);
    IF agid IS NOT NULL THEN
        RETURN(SELECT ab_ix FROM abk WHERE ab_plan_ag_id = agid AND ab_ap_nr = aknr ORDER BY ab_ix DESC LIMIT 1); -- mehrere PlanABK möglich, Problem #8863
    END IF;

    -- 3. versuche über Artikelnummer. Nur, wenn es nur eine Verbindung gibt
    FOR r IN SELECT ab_ix FROM abk JOIN auftg ON ag_id = ab_plan_ag_id WHERE ag_astat IN ('E', 'R', 'A') AND NOT ag_done AND ab_ap_nr = ldaknr LOOP -- ORDER BY Zufall? vgl. Problem #8863
        IF result IS NULL THEN
            result:= r.ab_ix; -- erster Datensatz, dieses Ergebnis sollte es sein
        ELSE
            RETURN null; -- das passiert, wenn es mehr als einen Datensatz gibt
        END IF;
    END LOOP;

    -- 4. PlanABK quer gefunden eintragen
    PERFORM disablemodified();
    UPDATE ldsdok SET ld_planabk = result WHERE ld_abk = abix AND ld_planabk IS DISTINCT FROM result;
    PERFORM enablemodified();

    RETURN result;
  END $$ LANGUAGE plpgsql; -- STABLE geht nicht wegen UPDATE ldsdok
--

--
DROP FUNCTION IF EXISTS tabk.abk__create; -- IMMER DROP wegen alter Version mit unterschiedlichen Signaturen!
CREATE OR REPLACE FUNCTION tabk.abk__create(
      _ab_ap_nr         varchar,
      _ab_askix         integer,
      _ab_st_uf1        numeric,
      _ab_st_uf1_soll   numeric = null,
      _parentabk        integer = null,
      _ld_id            integer = null,
      _ab_tablename     varchar = null,
      _ab_dbrid         varchar = null,
      _ab_stat          varchar = null,
      _ab_ap_bem        varchar = null,
      _pos              integer = null,
      _ab2_group__auto  boolean = true
  ) RETURNS integer AS $$
  DECLARE
      _opl    record;
      _ab_ix  integer;
  BEGIN
      -- ABK erstellen
        -- https://redmine.prodat-sql.de/projects/prodat-v-x/wiki/Abk
        -- https://redmine.prodat-sql.de/projects/prodat-v-x/wiki/ABK-Index+Strukturen


      -- Daten aus ASK vorbereiten
      SELECT
        -- Variantentext und Teilestatus
        op_txt, op_stat,
        -- Gemeinkosten-Zuschläge
        op_agk, op_mgk, op_rgk, op_fgk
      FROM opl
      WHERE op_ix = _ab_askix
      INTO _opl
      ;


      PERFORM TSystem.execution_code__disable('ak_copy_exactly__disabled');

      -- eigentlich ABK-Erstellung
      INSERT INTO abk (
          -- Daten aus Parametern
          ab_ap_nr,       ab_askix,     ab_st_uf1,    ab_st_uf1_soll,
          ab_parentabk,   ab_ld_id,
          ab_tablename,   ab_dbrid,
          ab_stat,        ab_ap_bem,    ab_pos,
          -- Daten aus ASK
          ab_txt,         ab_op_stat,
          ab_nk_mgk,
          ab_nk_agk,
          ab_nk_fgk,
          ab_nk_rgk
        )
      SELECT
          -- Parameter
          _ab_ap_nr,      _ab_askix,    _ab_st_uf1,   _ab_st_uf1_soll,
          _parentabk,     _ld_id,
          _ab_tablename,  _ab_dbrid,
          _ab_stat,       _ab_ap_bem,   _pos,
          -- ASK
          _opl.op_txt,    _opl.op_stat,
          -- Fallback auf System-Defaults für Gemeinkosten
            -- insb. bei Projekt-ABK ohne ASK (früher über ABK-Verbuchung per tplanterm.buchrm)
          coalesce( _opl.op_mgk, TSystem.Settings__GetNumeric('NK_MGK') ),
          coalesce( _opl.op_agk, TSystem.Settings__GetNumeric('NK_AGK') ),
          coalesce( _opl.op_fgk, TSystem.Settings__GetNumeric('NK_FGK') ),
          coalesce( _opl.op_rgk, TSystem.Settings__GetNumeric('NK_RGK') )
      RETURNING ab_ix
      INTO _ab_ix
      ;


      -- Bei vorhandener ASK und valider Menge: Arbeitsgänge, Materialien, Sondereinzelkosten und globale Zuschläge an ABK erzeugen
      IF _ab_askix IS NOT NULL AND _ab_st_uf1 > 0 THEN
          -- ABK-AG aus ASK übertragen
          PERFORM tabk.abk__ab2__from_op2__create( _ab_ix, _ab_askix ); -- generate_agfromavor


          -- ABK-Materialien aus ASK übertragen
          PERFORM tabk.abk__auftgi__from_op6__create( _ab_ix, _ab_askix, _ab_st_uf1 ); -- generate_rmatfromavor


          -- Sondereinzelkosten übertragen
          INSERT INTO nkz
            ( nz_ab_ix, nz_text,  nz_stk, nz_op3_ep )
          SELECT
              _ab_ix,   o3_txt,   1,      o3_preis
          FROM op3
          WHERE o3_ix = _ab_askix
          ;


          -- globale Zuschläge übertragen
          INSERT INTO nk7zko
            ( n7zk_ab_ix, n7zk_krc_bez, n7zk_proz )
          SELECT
              _ab_ix,     o7zk_krc_bez, o7zk_proz
          FROM op7zko
          WHERE o7zk_ix = _ab_askix
          ;


          -- verwendete Kostenstellen mit Stundensätzen übertragen (siehe auch nk2__a_iu)
          INSERT INTO nkksv (
              nkk_ab_ix,  nkk_ks,   -- ABK an Kostenstelle
              -- Norm- und Grenzstundensätze
              nkk_nssf,   nkk_gssf, -- Fertigung (Ausführung)
              nkk_nssr,   nkk_gssr, -- Rüsten
              nkk_nssm,   nkk_gssm  -- Bedienung (Personalzeit)
            )
          SELECT DISTINCT ON ( ks_abt )
              _ab_ix,     ks_abt,
              -- Norm- und Grenzstundensätze aus KS und überschriebener Stundensatz aus ASK
              coalesce(o2_sts, ks_sts), ks_gss,
              ks_stsr,    ks_gssr,
              ks_stsm,    ks_gssm
          FROM opl
            JOIN op2 ON o2_ix = op_ix
            JOIN ksv ON ks_abt = o2_ks
          WHERE op_ix = _ab_askix
          -- überschriebenen Stundensatz aus ASK bevorzugen, vgl. UNIQUE INDEX nkksv_index
          ORDER BY ks_abt, o2_sts IS NOT NULL DESC
          ;

      END IF;


      -- ASK-Prüfplan aktualisieren
      UPDATE op8 SET
        o8_unitsleft  = o8_unitsleft - _ab_st_uf1,
        o8_cyclesleft = o8_cyclesleft - 1
      WHERE o8_ix = _ab_askix
      ;

      PERFORM TSystem.execution_code__enable('ak_copy_exactly__disabled');

      IF _ab2_group__auto THEN
         PERFORM tabk.ab2_group__auto__by__ab_ix(_ab_ix);
      END IF;


      RETURN _ab_ix;
  END $$ LANGUAGE plpgsql;
--

-- Arbeitsgänge aus AVOR laden
CREATE OR REPLACE FUNCTION tabk.abk__ab2__from_op2__create(
    abix                INTEGER, -- An diese ABK
    askix               INTEGER, -- Arbeitsgänge aus dieser Stammkarte anhängen
    appendAG            BOOLEAN DEFAULT FALSE, -- Arbeitsgänge sollen am Ende angehangen werden
    projStruParentTable VARCHAR DEFAULT NULL,  -- Wenn ABK in Projektstrukturframe hängt, kann hier die Haupttabelle angegeben werden,
                                               -- um ggf. auf unterschiedliche Weise behandelt zu werden
    ask_ag              INTEGER DEFAULT NULL -- nur bestimmten AG einfügen (Vgl. tplanterm.refresh_abkabg)
  ) RETURNS VOID AS $$
  DECLARE r RECORD;
          a2n INTEGER;
          a2id INTEGER;
  BEGIN
    FOR r IN SELECT abix, o2_ks, o2_ksap, o2_v_ll_dbusename, o2_aw, o2_awpreis, o2_preis_table, o2_preis_dbrid, o2_id, o2_n, o2_nc, 1, o2_zeinh_tr,
                    o2_zeinh_tx, o2_tr, o2_th, o2_tm, o2_tn, o2_ul, o2_lgz, o2_dlz, o2_tv, o2_maschauto_ta, o2_txt, o2_txt_rtf,
                    o2_txt_intern, o2_aknr, o2_awtx, o2_adkrz, o2_msp_id
             FROM op2 JOIN opl ON op_ix = o2_ix
             WHERE o2_ix = askix
               AND (ask_ag IS NULL OR o2_n = ask_ag) -- nur bestimmter AG
             ORDER BY o2_n
    LOOP
        --Arbeitsgänge hinten anhängen
        IF appendAG THEN
            a2n:= max(a2_n) + 10 FROM ab2 WHERE a2_ab_ix = abix;
            a2n:= COALESCE(a2n, 10);
        ELSE
            a2n:= r.o2_n;
        END IF;

        -- Entfernt, die Betreff-aus-AG-Text soll erhalten bleiben. Kann nach Kenntnissnahme raus.
        -- CASE WHEN (LOWER(COALESCE(projStruParentTable,''))='qab') THEN subject:=''; ELSE subject:=r.o2_txt::VARCHAR(50); END CASE;

        INSERT INTO ab2 (a2_ab_ix, a2_ks, a2_ksap, a2_v_ll_dbusename, a2_ausw, a2_awpreis, a2_o2_id, a2_n,
                         a2_ncnr, a2_prio, a2_zeinh_tr, a2_zeinh_tx, a2_tr, a2_th, a2_tm, a2_tn, a2_ul, a2_lgz, a2_dlz,
                         a2_tv, a2_maschauto_ta, a2_subject, a2_txt, a2_txt_rtf, a2_txt_intern, a2_aknr, a2_awtx, a2_adkrz,
                         a2_preis_table, a2_preis_dbrid, a2_msp_id )
        VALUES (abix, r.o2_ks, r.o2_ksap, r.o2_v_ll_dbusename, r.o2_aw, r.o2_awpreis, r.o2_id, a2n,
                r.o2_nc, 1, r.o2_zeinh_tr, r.o2_zeinh_tx, r.o2_tr, r.o2_th, r.o2_tm, r.o2_tn, r.o2_ul, r.o2_lgz, r.o2_dlz,
                r.o2_tv, r.o2_maschauto_ta, r.o2_txt::VARCHAR(50), r.o2_txt, r.o2_txt_rtf, r.o2_txt_intern, r.o2_aknr,
                r.o2_awtx, r.o2_adkrz, r.o2_preis_table, r.o2_preis_dbrid, r.o2_msp_id)
        RETURNING a2_id INTO a2id;

        --umkopieren op2ba in ab2ba
        INSERT INTO ab2ba (a2ba_ak_nr, a2ba_a2_id, a2ba_noz_id, a2ba_txt, a2batxt_rtf)
        SELECT             o2ba_ak_nr,       a2id, o2ba_noz_id, o2ba_txt, o2batxt_rtf
        FROM op2ba
        WHERE o2ba_o2_id = r.o2_id;
    END LOOP;

    RETURN;
  END $$ LANGUAGE plpgsql;
--

-- Material aus AVOR laden
CREATE OR REPLACE FUNCTION tabk.abk__auftgi__from_op6__create(
      _abkix  integer,
      _askix  integer,
      _menge  numeric
  ) RETURNS void AS $$
  DECLARE
      _op6                  record;
      _agid                 integer;
      _beistellung_kunde    boolean;
      _fertigungslos_faktor numeric;  -- Faktor bzgl. Fertigungsmenge und Los
      _soll_menge           numeric;  -- Menge des Materials (ag_stk)
      _zuschnitt_menge      numeric;  -- Zuschnittsmenge des Materials (ag_o6_stkz)
      _nk_menge             numeric;  -- Menge geliefert in Nachkalkulation (ag_nk_stkl)
  BEGIN
    -- Hinweis: bei Projektbezug generiert sich die AUFTG die Auftragsnummer selbst. siehe auftg__b_i
    FOR _op6 IN

        SELECT
              o6_aknr, o6_dimi, o6_calcOnly, o6_stkz, o6_lz, o6_bz, o6_hz, o6_zz, o6_mce, o6_zme, o6_m, o6_m_stat,
              o6_o2_n,
              o6_ekenner_krz, o6_artpr, o6_mgk, o6_stat, o6_pos,
              op_lg, ak_norm
         FROM op6
         JOIN art ON ak_nr = o6_aknr
         JOIN opl ON op_ix = o6_ix
        WHERE o6_ix = _askix
          -- Rohmaterialien, welche selbst Vorfertigung sind, ausschliessen
          AND NOT EXISTS(SELECT true FROM stv WHERE st_zn = o6_aknr)
          AND NOT EXISTS(SELECT true FROM opl WHERE op_n = o6_aknr AND op_standard)
          -- alternative MatPos herausfiltern, siehe #1538
/*STATUS#20827...*/
          AND NOT TSystem.ENUM_ContainsValue( o6_stat, 'KS,AM') --nur Konstruktionsinfo, Alternativmaterial
              -- TODO KS ist hier mMn falsch. KS darf nur nicht weiter aufgelöst werden aber selbst als Artikel muss es aufgenommen werden!

    LOOP
        _fertigungslos_faktor := null;
        _soll_menge := null;
        _zuschnitt_menge := null;
        _nk_menge := null;


        -- Faktor auf Materialmenge (pro Los oder pro Fertigungsteil) bestimmen.
        -- Menge pro Los
        IF _op6.o6_m_stat = 1 THEN

            -- Losfaktor auf volle Lose aufrunden, Ticket: #20081
            _fertigungslos_faktor := ceil(_menge / _op6.op_lg);

            -- Menge pro Los, aber Los = 1, dann Menge einmalig (Kreyenberg)
            IF _op6.op_lg = 1 THEN
                _fertigungslos_faktor := 1;
            END IF;

        -- Menge nicht pro Los, sondern pro Fertigungsteil
        ELSE
            _fertigungslos_faktor := _menge;
        END IF;


        -- Mengen bestimmen
        -- für bloße ASK-Kalkulationspositionen
        IF _op6.o6_calcOnly THEN

            _soll_menge := 0;
            _zuschnitt_menge := 0;

            -- Für bloße ASK-Kalkulationsposition NK-Menge als vollständig geliefert kennzeichnen (uf1).
            _nk_menge := tartikel.me__menge__in__menge_uf1( _op6.o6_mce, _op6.o6_m ) * _fertigungslos_faktor;

        -- für fertigungsrelevante Positionen
        ELSE

            -- Menge des Materials anhand Fertigungslos
            _soll_menge := _op6.o6_m * _fertigungslos_faktor;

            -- Zuschnittsmenge des Materials anhand Fertigungsmenge
            _zuschnitt_menge := _op6.o6_stkz * _menge;

            -- leer für Anwendereingabe
            _nk_menge := null;

        END IF;


        -- ABK-Materialposition aus ASK-Material anlegen
        INSERT INTO auftg (
            ag_post2,     ag_parentabk, ag_astat,   ag_stat,      ag_aknr,      ag_mcv,
            ag_a2_id,
            ag_o6_lz,     ag_o6_bz,     ag_o6_hz,   ag_o6_zz,     ag_o6_zme,    ag_o6_dimi,   ag_o6_pos,   ag_o6_stkz,
            ag_stk,       ag_nk_stkl,
            ag_vkp
          )
        SELECT
            'R',          _abkix,       'I',        _op6.o6_stat, _op6.o6_aknr, _op6.o6_mce,
            (SELECT a2_id FROM ab2 WHERE a2_ab_ix = _abkix AND a2_n = _op6.o6_o2_n),
            _op6.o6_lz,   _op6.o6_bz,   _op6.o6_hz, _op6.o6_zz,   _op6.o6_zme,  _op6.o6_dimi, _op6.o6_pos, _zuschnitt_menge,
            _soll_menge,  _nk_menge,
            _op6.o6_artpr * ( 1 + coalesce( _op6.o6_mgk, 0 ) / 100 )

        RETURNING ag_id INTO _agid
        ;


        -- Status aus op6 in auftgmatinfo übernehmen ; ACHTUNG: analog TAuftg.AppendIntAuftg
        _beistellung_kunde := TSystem.ENUM_GetValue( _op6.o6_stat, 'BK');

        IF      _beistellung_kunde
            OR  _op6.o6_ekenner_krz IS NOT NULL
        THEN

           -- Beistellung durch Kunden markieren
           UPDATE auftgmatinfo
              SET agmi_beistell     = _beistellung_kunde,
                  agmi_ekenner_krz  = _op6.o6_ekenner_krz
            WHERE agmi_ag_id = _agid
            ;

        END IF;

    END LOOP;


    RETURN;
  END $$ LANGUAGE plpgsql;

  --DROP
  --CREATE OR REPLACE FUNCTION tabk.generate_rmatfromavor(abkix INTEGER, askix INTEGER, agnr VARCHAR, Menge NUMERIC, Norm VARCHAR DEFAULT NULL) RETURNS VOID AS $$
  --  SELECT tabk.abk__auftgi__from_op6__create(abkix, askix, agnr, Menge, Norm);
  -- $$ LANGUAGE sql;
--

CREATE OR REPLACE FUNCTION tabk.abk__auftgi__stv__create(
    IN _ParentAbk   integer,
    IN _MainAbk     integer,
    IN _aknr        varchar,
    IN _stk         numeric,
    IN _stvtrs_dbrid  varchar = null,
    IN _ststat        varchar = null,
    IN _stekenner     varchar = null,
    IN _KontaktAP     varchar = null
    )
    RETURNS integer
    AS $$
    DECLARE _result integer;
            _Beistellung_Kunde boolean;
    BEGIN

      PERFORM TSystem.execution_code__disable('ak_copy_exactly__disabled');

      INSERT INTO auftg
                  (ag_astat, ag_parentabk, ag_mainabk, ag_aknr, ag_stk, ag_kontakt, ag_stat)
           VALUES ('I',      _ParentAbk,   _MainAbk,   _aknr,   _stk,   _KontaktAP, _ststat)
        RETURNING ag_id
             INTO _result
      ;

      PERFORM TSystem.execution_code__enable('ak_copy_exactly__disabled');

      --Hinweis: Beistellung im Trigger "auftg__a_i"

      -- stvtrs gibt es nur bei Stücklistenauflösung. Nicht zB bei Beistellmaterial
      IF _stvtrs_dbrid IS NOT null THEN
        UPDATE stvtrs SET auftg_ag_id = _result WHERE dbrid = _stvtrs_dbrid;
      END IF;

      -- ACHTUNG: Code analog tabk.abk__auftgi__from_op6__create
      _Beistellung_Kunde := coalesce( _ststat = 'BK', false );

      IF    _Beistellung_Kunde
         OR ( _stekenner IS NOT null )
      THEN
        UPDATE auftgmatinfo SET agmi_beistell = _Beistellung_Kunde, agmi_ekenner_krz = _stekenner WHERE agmi_ag_id = _result;
      END IF;

      RETURN _result;

END $$ LANGUAGE plpgsql;

--abk hat offene AG?  -- http://redmine.prodat-sql.de/projects/prodat-v-x/wiki/Abk
CREATE OR REPLACE FUNCTION tabk.abk__ab2__a2_ende__openag(abix integer) RETURNS boolean AS $$ --DROP
   SELECT a2_ende FROM ab2 WHERE a2_ab_ix = abix ORDER BY a2_ende LIMIT 1; --true, false, null
   $$ LANGUAGE sql STABLE PARALLEL SAFE STRICT;

--abk hat offenes mat  -- http://redmine.prodat-sql.de/projects/prodat-v-x/wiki/Abk
CREATE OR REPLACE FUNCTION tabk.abk__auftgi__ag_done__openmat(abix integer) RETURNS boolean AS $$
    SELECT ag_done FROM auftg WHERE ag_parentabk = abix ORDER BY ag_done LIMIT 1; --true, false, null
   $$ LANGUAGE sql STABLE PARALLEL SAFE STRICT;

--
-- Status, ob alle Bedarfe der ABK voll geliefert bzw. erledigt sind, s. #7583.
-- Optionen: inkl. Unter-ABKs (komplette Fertigungsstruktur); nur Material-Bedarfe: (R)oh- und (N)ormteile oder alles (auch Unterbauteile)
-- abk__auftgi__ag_done__openmat ??
CREATE OR REPLACE FUNCTION tabk.abk_bedarf_voll_geliefert(
    IN in_abix integer,
    IN include_childs boolean DEFAULT true,
    IN only_mat boolean DEFAULT false,
    IN _child_abk_array integer[] DEFAULT '{}'
    )
    RETURNS boolean AS $$
    DECLARE abix_arr integer[];
    BEGIN

      IF include_childs THEN
        IF _child_abk_array = '{}' THEN
          abix_arr := array_agg(get_all_child_abk) FROM tplanterm.get_all_child_abk(in_abix);
        ELSE
          abix_arr := _child_abk_array;
        END IF;
      ELSE
        abix_arr := array_append(null, in_abix);
      END IF;

      RETURN coalesce((
          SELECT bool_and(ag_stk_uf1 <= ag_stkl OR ag_done) -- voll geliefert oder Position ist erfüllt
            FROM auftg
           WHERE ag_astat = 'I'
             AND NOT ag_storno -- stornierte sind uninteressant
             AND ag_parentabk = ANY (abix_arr)
             AND (ag_post2 IN ('R', 'N') OR NOT only_mat) -- nur Rohmaterial und Normteile oder alles
          ), true) -- nichts gefunden? Dann gilt das als Bedarf voll geliefert.
      ;
    END $$ LANGUAGE plpgsql STABLE PARALLEL SAFE STRICT;

CREATE OR REPLACE FUNCTION tabk.abk_bedarf_vg_mat_childs(
    IN in_abix integer,
    IN include_childs boolean DEFAULT true,
    IN only_mat boolean DEFAULT false
    )
    RETURNS boolean
    AS $$
      SELECT tabk.abk_bedarf_voll_geliefert(in_abix, include_childs, only_mat);
    $$ LANGUAGE sql;
--

-- Gibt es noch offene, nicht lieferfähige Unterbauteile für eine ABK
-- http://redmine.prodat-sql.de/projects/prodat-v-x/wiki/Abk
CREATE OR REPLACE FUNCTION tabk.abk__auftgi__ak_tot__fehlteile(IN _abix integer) RETURNS boolean AS $$
  SELECT EXISTS(SELECT true
                  FROM auftg
                  JOIN art ON ak_nr = ag_aknr
                 WHERE ag_parentabk = _abix
                   AND NOT ag_done
                   AND NOT TSystem.ENUM_ContainsValue(ag_stat, 'UE')
                   AND ak_lag
                   AND ak_tot < ag_stk_uf1 - ag_stkl
                );
  $$ LANGUAGE sql STABLE PARALLEL SAFE STRICT;

CREATE OR REPLACE FUNCTION tabk.abk__auftgi__bedarf__getbestand__fehlteile(IN _abix integer) RETURNS boolean AS $$
  SELECT coalesce(
               ( -- array agg // EXISTS ????? tabk.abk__auftgi__bedarf__getbestand__fehlteile__ak_nr__list(_abix) IS NOT null -- irgendein Artikel ist in Bedarsberechnung zum Termin < 0
                SELECT min( tartikel.bedarf__getbestand(ag_aknr, coalesce(ag_ldatum, ag_kdatum)) ) < 0--hier bin ich selbst berücksichtigt und bereits abgezogen
                  FROM auftg
                  JOIN art ON ak_nr = ag_aknr
                  LEFT JOIN auftgmatinfo ON agmi_ag_id = ag_id AND NOT agmi_beistell
                 WHERE ag_parentabk = _abix
                   AND NOT ag_done
                   AND NOT TSystem.ENUM_ContainsValue(ag_stat, 'UE')
                   AND ak_lag
                   -- AND ag_ownabk IS NULL
                )
          , false
          )
  $$ LANGUAGE sql STABLE STRICT PARALLEL SAFE; -- PARALLEL SAFE?? => unklar, siehe bedarf__getbestand!

CREATE OR REPLACE FUNCTION tabk.abk__auftgi__bedarf__getbestand__fehlteile__ak_nr__list(IN _abix integer) RETURNS SETOF varchar AS $$
                SELECT array_agg(ag_aknr)
                  FROM auftg
                  JOIN art ON ak_nr = ag_aknr
                  LEFT JOIN auftgmatinfo ON agmi_ag_id = ag_id AND NOT agmi_beistell
                 WHERE ag_parentabk = _abix
                   AND NOT ag_done
                   AND NOT TSystem.ENUM_ContainsValue(ag_stat, 'UE')
                   AND ak_lag
                   AND /*min*/( tartikel.bedarf__getbestand(ag_aknr, coalesce(ag_ldatum, ag_kdatum)) ) < 0--hier bin ich selbst berücksichtigt und bereits abgezogen
                   -- AND ag_ownabk IS NULL
  $$ LANGUAGE sql STABLE STRICT PARALLEL SAFE; -- PARALLEL SAFE?? => unklar, siehe bedarf__getbestand!
--

CREATE OR REPLACE FUNCTION tabk.abk__ab2__a2_id__ks_plan__by__ab_ix(_ab_ix integer) RETURNS integer
  AS $$
    SELECT a2_id FROM ab2 JOIN ksv ON a2_ks = ks_abt WHERE a2_ab_ix = _ab_ix AND NOT a2_ende AND ks_plan ORDER BY a2_n LIMIT 1;
  $$ LANGUAGE sql STABLE PARALLEL SAFE STRICT;

--

-->>CREATE OR REPLACE FUNCTION tplanterm.get_all_child_abk
--
CREATE OR REPLACE FUNCTION tabk.abk__auftgi__setartikel__get_parent__ld_abk(parentABK integer) RETURNS integer AS $$ --zu einem set den übergeordneten Fertitgungsartikel
    DECLARE rec record;
    BEGIN
     --laufe solange nach oben, bis die ABK kommt, welche eine interne Bestellung hat
     SELECT ab_ld_id, ab_ix, ab_parentABK INTO rec FROM abk WHERE ab_ix=parentABK;
     IF rec.ab_ld_id IS NOT NULL THEN
        RETURN rec.ab_ix;
     ELSIF rec.ab_parentABK IS NOT NULL THEN
        RETURN tabk.abk__auftgi__setartikel__get_parent__ld_abk(rec.ab_parentABK);
     ELSE
        RETURN null;
     END IF;
    END $$ LANGUAGE plpgsql STABLE STRICT;
 --
CREATE OR REPLACE FUNCTION tabk.abk__auftgi__setartikel__get_all_child_setabk(integer) RETURNS SETOF integer AS $$
    DECLARE rec record;
    BEGIN
     --Gib alle ABK zurück, welche selbst keine interne Bestellung haben und somit ABK-Struktur/Set sind
     FOR rec IN SELECT DISTINCT get_all_child_abk FROM tplanterm.get_all_child_abk($1, False, True) LOOP
                RETURN NEXT rec.get_all_child_abk;
     END LOOP;
     RETURN;
    END $$ LANGUAGE plpgsql STABLE STRICT;
 --
CREATE OR REPLACE FUNCTION tabk.abk__auftgi__parentabk__aknr__bez(parentABK integer) RETURNS varchar(250) AS $$
 DECLARE rec record;
         result_bez varchar(250);
 BEGIN
  SELECT ab_ix, ab_parentABK, ab_mainABK, ab_ap_nr, ab_ap_bem, ld_aknr
    INTO rec
    FROM abk
    LEFT JOIN ldsdok ON ld_abk = ab_ix
   WHERE ab_ix = parentABK;
  --
   SELECT coalesce(rec.ab_ap_nr, ak_nr, '') || coalesce(' ~ ' || coalesce(rec.ab_ap_bem, ak_bez, ''))
     INTO result_bez
     FROM art
    WHERE ak_nr = coalesce(rec.ab_ap_nr, rec.ld_aknr);
  --
   RETURN    coalesce(lpad(parentABK, 6, ' ') || ' :','')
          || coalesce((SELECT CHR(65+stvtrs.ebene) ||'.'|| lpad(stvtrs.pos, 5, 0)
                         FROM stvtrs
                        WHERE resid = 'ldsdok_ld_id_' || (SELECT ld_id
                                                            FROM ldsdok
                                                           WHERE ld_abk = rec.ab_mainabk
                                                         )
                          AND stvtrs.id = (SELECT DISTINCT ag_post1
                                             FROM auftg a1
                                            WHERE a1.ag_ownabk = parentABK LIMIT 1
                                           )
                       ) ||  ' ', 'A.00000 '
                      ) --stücklistenposition übergeordneter Artikel
          || coalesce(result_bez, '');
 END $$ LANGUAGE plpgsql STABLE;
--
CREATE OR REPLACE FUNCTION tabk.abk__ldsdok__term_set(abix integer) RETURNS BOOL AS $$
  BEGIN
    UPDATE ldsdok
       SET ld_terml = ab_et, ld_termweekl = NULL
      FROM abk
     WHERE ab_ix = abix
       AND ld_abk = ab_ix
       AND ld_terml IS DISTINCT FROM ab_et;

    --Nur ld_terml setzen. Datum geplant bleibt erhalten, ist idR Kundenwunschtermin. In Plantafel werde alle die als unterminiert angezeigt, welche kein ld_terml haben.
    --2015-05-07: Blödsinn mit Wunschtermin fällt weg. Wenn terminiert, steht in beiden Terminen (auch ld_term) das Endedatum. Dadurch stimmt Termin ABK "Werkstatttermin" mit dem der Terminierung überein.
    --2015-07-09: Allerdings kann ld_term per Setting erhalten bleiben. #6290
    IF NOT TSystem.Settings__GetBool('ABK.Datum.geplant.erhalten') THEN --Hinweis: wird auch als Setting genutzt um Kreisaufruf bei automatischer Terminierung zu vermeiden > ldsdok__a_90_iu__airbus_loll
       UPDATE ldsdok
          SET ld_term = ab_et, ld_termweek = NULL
         FROM abk
        WHERE ab_ix = abix
          AND ld_abk = ab_ix
          AND ld_term IS DISTINCT FROM ab_et;
    END IF;
    RETURN found;
  END $$ LANGUAGE plpgsql;

-- die alte Funktion: abk__qab_get(abix integer) RETURNS integer
-- Ermittlung vor- sowie nachgelagerter ABKs sowie QABs zu einer ABK
CREATE OR REPLACE FUNCTION TAbk.abk__qab_abk__get(in_abix IN integer) RETURNS TABLE(sort   integer,
                                                                                   objekt varchar,
                                                                                   abix   integer,
                                                                                   qnr    integer) AS $$

-- #12763 - gibt alle vorgelagerten und Nachgelagerten QABs und ABKs zurück
-- sort = 1 -- ggf. ein Datensatz für vorgelagerte ABK (also eine ABK bei der ein QAB ausgelöst wurde und ich (in_abix) die Nacharbeits-ABK bin; wartet auf diese ABK)
-- sort = 2 -- ggf. ein Datensatz für vorgelagerte QAB (also ein QAB der auf eine ABK ausgelöst wurde und ich (in_abix) bin die Nachbarbeits-ABK darauf; wartet auf diese ABK)
-- sort = 3 -- 1:n nachgelagerte QABs (also Qualitätsvorfälle, die wegen mir (in_abix) aufgetreten sind; ich (in_abix) warte auf diese QABs)
-- sort = 4 -- 1:n nachgelagerte ABKs (also Nacharbeits-ABKs, die wegen Qualitätsvorfällen bei mir (in_abix) aufgetreten sind; ich (in_abix) warte auf diese ABKs)
BEGIN
  IF in_abix IS NULL THEN
    RETURN next;
  END IF;

 RETURN QUERY
   -- ABK vorgelagerte
  SELECT DISTINCT ON (q_ab_ix)  1 AS sort,
                                'ABK'::varchar as objekt,
                                q_ab_ix AS ab_ix,
                                NULL::integer as  q_nr
  FROM abk
    LEFT JOIN qab ON qab.dbrid = ab_dbrid
  WHERE ab_tablename = 'qab'
    AND ab_ix = in_abix
    AND q_ab_ix IS  NOT NULL
  UNION
  -- QAB vorgelagert
  SELECT DISTINCT ON (q_nr) 2 AS sort,
                            'QAB'::varchar AS objekt,
                            NULL::integer AS ab_ix,
                            q_nr
  FROM abk
    LEFT JOIN qab ON qab.dbrid = ab_dbrid
  WHERE ab_tablename = 'qab'
      AND ab_ix = in_abix
  UNION
   -- QAB nachgelagerte
  SELECT DISTINCT on (q_nr) 3 AS sort,
                            'QAB'::varchar AS objekt,
                            NULL::integer AS ab_ix,
                            q_nr
  FROM abk
    LEFT JOIN qab ON q_ab_ix = ab_ix
  WHERE q_nr IS NOT NULL
    AND ab_ix = in_abix
  UNION
  -- ABK nachgelagert
  SELECT 4 AS sort,
        'ABK'::varchar,
        ab_ix,
        NULL::integer AS q_nr
  FROM abk
    LEFT JOIN qab ON qab.dbrid = ab_dbrid
  WHERE ab_tablename = 'qab'
    AND q_ab_ix = in_abix
  ORDER BY sort, ab_ix, q_nr;
END $$ LANGUAGE plpgsql STRICT;

CREATE OR REPLACE FUNCTION TAbk.abk__qab_abk__get_compressed(in_abix IN INTEGER) RETURNS TABLE(typ VARCHAR,
                                                                                               abix   INTEGER,
                                                                                               qnr    INTEGER) AS $$
-- #12763 - gibt alle vorgelagerten und Nachgelagerten QABs und ABKs zurück
-- vorgelagert: ABK (also eine ABK bei der ein QAB ausgelöst wurde und ich (in_abix) die Nacharbeits-ABK bin; wartet auf diese ABK) und
--              QAB (also ein QAB der auf eine ABK ausgelöst wurde und ich (in_abix) bin die Nachbarbeits-ABK darauf; wartet auf diese ABK)
-- nachgelagert: 1:n nachgelagerte QABs (also Qualitätsvorfälle, die wegen mir (in_abix) aufgetreten sind; ich (in_abix) warte auf diese QABs) sowie
--               1:n nachgelagerte ABKs (also Nacharbeits-ABKs, die wegen Qualitätsvorfällen bei mir (in_abix) aufgetreten sind; ich (in_abix) warte auf diese ABKs)

BEGIN
  IF in_abix IS NULL THEN
    RETURN next;
  END IF;


 RETURN QUERY
  SELECT '1 vorgelagert'::VARCHAR, q_ab_ix, q_nr
  FROM abk LEFT JOIN qab ON qab.dbrid = ab_dbrid
  WHERE ab_tablename = 'qab'
  AND ab_ix = in_abix

  UNION

  SELECT '2 nachgelagert'::VARCHAR, a.ab_ix, q.q_nr
  FROM (
    (SELECT DISTINCT q_nr
     FROM abk
     LEFT JOIN qab ON q_ab_ix = ab_ix
     WHERE q_nr IS NOT NULL
     AND ab_ix = in_abix) q

     LEFT JOIN

     (SELECT ab_ix
      FROM abk
      LEFT JOIN qab ON qab.dbrid = ab_dbrid
      WHERE ab_tablename = 'qab'
      AND q_ab_ix = in_abix) a on true) ;

END $$ LANGUAGE plpgsql STRICT;




-- Ermittelt die QAB mit Nacharbeit, für den diese ABK erstellt wurde. Nur ein Datensatz/Letzter
CREATE OR REPLACE FUNCTION TAbk.abk__qab_get(abix INTEGER) RETURNS INTEGER AS $$
DECLARE
 rec record;
BEGIN
  FOR rec IN SELECT *
              FROM TAbk.abk__qab_abk__get(in_abix => abix)
              WHERE qnr is not null
              ORDER by sort DESC, abix DESC, qnr DESC LOOP
    RETURN rec.qnr;
  END LOOP;
  RETURN null;
END $$ LANGUAGE plpgsql STABLE;

CREATE OR REPLACE FUNCTION TAbk.abk__ab_stat__folgeabk__is(_abk abk) RETURNS boolean
  AS $$
    SELECT TSystem.ENUM_ContainsValue(_abk.ab_stat, 'F-ABK') IS true;
  $$ LANGUAGE sql STABLE PARALLEL SAFE;
CREATE OR REPLACE FUNCTION TAbk.abk__ab_stat__folgeabk__is(_abix integer) RETURNS boolean
  AS $$
    SELECT (SELECT TAbk.abk__ab_stat__folgeabk__is(abk) FROM abk WHERE ab_ix = _abix) IS true;
  $$ LANGUAGE sql STABLE PARALLEL SAFE STRICT;

-- AB-Status / TYP als anzeigbarer Text
CREATE OR REPLACE FUNCTION TAbk.abk__ab_stat__descr(IN ab_stat VARCHAR(3)) RETURNS VARCHAR(100) AS $$
DECLARE
  descr VARCHAR(100);
BEGIN
  IF    TSystem.ENUM_ContainsValue(ab_stat, 'PL')         THEN descr := lang_text(25234);  -- Plan-ABK https://redmine.prodat-sql.de/projects/prodat-v-x/wiki/Plantafel
  ELSIF TSystem.ENUM_ContainsValue(ab_stat, 'PS')         THEN descr := lang_text(13809);  -- Set-Artikel
  ELSIF TSystem.ENUM_ContainsValue(ab_stat, 'BGr-STD')    THEN descr := lang_text(35147);  -- Standard-Baugruppe aus Stücklistenauflösung
  ELSIF TSystem.ENUM_ContainsValue(ab_stat, 'QD')         THEN descr := lang_text(13810);  -- QS-Organisation
  ELSIF TSystem.ENUM_ContainsValue(ab_stat, 'QD_')        THEN descr := lang_text(13811);  -- QS-Aufgaben  QD2..QD7 für die einzelnen Register. Der Unterstrich ist Platzhalter für 1 einzelnes Zeichen.
  ELSIF TSystem.ENUM_ContainsValue(ab_stat, 'QNA')        THEN descr := lang_text(13064);  -- Nacharbeits-ABK
  ELSIF TSystem.ENUM_ContainsValue(ab_stat, 'QSA')        THEN descr := lang_text(13065);  -- Service-ABK
  ELSIF TSystem.ENUM_ContainsValue(ab_stat, 'QAB')        THEN descr := lang_text(26397);  -- QAB-Auslöser
  ELSIF TSystem.ENUM_ContainsValue(ab_stat, 'SPL')        THEN descr := lang_text(13817);  -- Splitt-ABK
  ELSIF TSystem.ENUM_ContainsValue(ab_stat, 'F-ABK')      THEN descr := lang_text(21095);  -- Folge-ABK
  ELSIF TSystem.ENUM_ContainsValue(ab_stat, 'BL')         THEN descr := lang_text(21093);  -- Beistellung an Lieferant
  ELSIF TSystem.ENUM_ContainsValue(ab_stat, 'DM')         THEN descr := lang_text(26566);  -- Demontage
  ELSIF TSystem.ENUM_ContainsValue(ab_stat, 'UB')         THEN descr := lang_text(26581);  -- Umbau
  ELSIF TSystem.ENUM_ContainsValue(ab_stat, 'UG')         THEN descr := lang_text(26582);  -- Upgrade
  ELSIF TSystem.ENUM_ContainsValue(ab_stat, 'AM')         THEN descr := lang_text(26332);  -- Änderungsmanagement
  ELSIF NullIf(ab_stat,'') IS NULL THEN descr := lang_text(13814);  -- Standard
  ELSE  descr := ab_stat||'=?';
  END IF;

  RETURN descr;

END $$ LANGUAGE plpgsql STABLE;


-- Ausschuss
    -- HINWEIS SETTING: Ausschluss von KS: TSystem.Settings__Set('Keine.Ausschuss-Steuerung.fuer.KS', '') kommasepariert
    -- Gibt kleinste gemeldete Fertigungsmenge der ABK bzw. Summe eines bestimmten AG zurück
    CREATE OR REPLACE FUNCTION tabk.get_bdea_fert(IN in_abix INTEGER, IN in_agende BOOLEAN DEFAULT NULL, IN in_ag INTEGER DEFAULT NULL) RETURNS NUMERIC(12,4) AS $$
      BEGIN
        RETURN COALESCE(MIN(fertmenge), 0) FROM ( -- kleinste Fertigungsmenge der Arbeitsgänge
            SELECT SUM(ba_stk) AS fertmenge -- Summe pro Arbeitsgang
            FROM bdea
            LEFT JOIN ab2 ON a2_ab_ix = ba_ix AND a2_n = ba_op
            WHERE ba_ix = in_abix AND ba_op = COALESCE(in_ag, ba_op) -- bestimmter AG oder alle
              AND CASE WHEN     in_agende THEN COALESCE(a2_ende, ba_ende)       -- in_agende-Status ist dreiwertig: beendete (true), offene (false), alle (NULL)
                       WHEN NOT in_agende THEN NOT COALESCE(a2_ende, ba_ende)   -- Arbeitsgang (ab2) beendet ist führend (Abweichungen zu bdea vorhanden),
                       ELSE true                                                -- sowie Status der Auftragszeiten (bdea) ohne Arbeitsgang (ab2) berücksichtigen.
                  END
              AND ba_ks NOT IN (SELECT regexp_split_to_table(TSystem.Settings__Get('Keine.Ausschuss-Steuerung.fuer.KS'), E'\\s*,\\s*')) -- Ausschluss von KS
            GROUP BY ba_op
            -- Auswärstrücklieferungen
            UNION
            SELECT SUM(ld_stkl) -- Summe der (Teil-)Rücklieferungen pro Arbeitsgang
            FROM ldsdok
            JOIN ab2 ON a2_id = ld_a2_id
            WHERE NOT ld_storno
              AND ld_stkl > 0 -- da Vergabe mit 0 angelegt wird
              AND a2_ab_ix = in_abix AND a2_n = COALESCE(in_ag, a2_n) -- bestimmter AG oder alle
              AND CASE WHEN     in_agende THEN a2_ende      -- in_agende-Status ist dreiwertig: beendete (true), offene (false), alle (NULL)
                       WHEN NOT in_agende THEN NOT a2_ende
                       ELSE true
                  END
              AND a2_ks NOT IN (SELECT regexp_split_to_table(TSystem.Settings__Get('Keine.Ausschuss-Steuerung.fuer.KS'), E'\\s*,\\s*')) -- Ausschluss von KS
            GROUP BY a2_n) AS sum_ba_stk_pro_ag;
      END $$ LANGUAGE plpgsql STABLE;
    --

    -- Gibt den Ausschuss einer ABK bzw. eines best. AG zurück
    CREATE OR REPLACE FUNCTION tabk.get_bdea_ausschuss(IN in_abix INTEGER, IN in_agende BOOLEAN DEFAULT NULL, IN in_ag INTEGER DEFAULT NULL, IN in_dat DATE DEFAULT current_date) RETURNS NUMERIC(12,4) AS $$
      BEGIN
        RETURN COALESCE((SELECT SUM(ba_auss) FROM bdea
                         LEFT JOIN ab2 ON a2_ab_ix = ba_ix AND a2_n = ba_op
                         WHERE ba_ix = in_abix AND ba_op = COALESCE(in_ag, ba_op) -- bestimmter AG oder alle
                           AND CASE WHEN in_agende     THEN COALESCE(a2_ende, ba_ende)       -- in_agende-Status ist dreiwertig: beendete (true), offene (false), alle (NULL)
                                    WHEN NOT in_agende THEN NOT COALESCE(a2_ende, ba_ende)   -- Arbeitsgang (ab2) beendet ist führend (Abweichungen zu bdea vorhanden),
                                    ELSE true                                                -- sowie Status der Auftragszeiten (bdea) ohne Arbeitsgang (ab2) berücksichtigen.
                               END
                           AND ba_ks NOT IN (SELECT regexp_split_to_table(TSystem.Settings__Get('Keine.Ausschuss-Steuerung.fuer.KS'), E'\\s*,\\s*')) -- Ausschluss von KS
                           AND ba_anf::DATE <= in_dat -- nach Stichtag
                         ), 0);
      END $$ LANGUAGE plpgsql STABLE;
--

-- Berechnung und Eintrag der ABK-Eigenschaft ('System.Freigabemenge.Unterproduktion') zur Freigabe ('System.Quittierung.Unterproduktion') der zu gering produzierten Menge.
-- kleinste gemeldete Fertigungsmenge geschlossener AGs - Ausschüsse der offenen AGs < Sollmenge
CREATE OR REPLACE FUNCTION tabk.set_abkrecno_unterproduktion(IN in_abix INTEGER) RETURNS VOID AS $$
  DECLARE abk_dbrid VARCHAR;
          sollMenge NUMERIC(12,4);      -- Sollmenge
          fertMenge NUMERIC(12,4);      -- Fertigungsmenge
          effektiveMenge NUMERIC(12,4); -- Fertigungsmenge abzgl. der aktuellen Ausschüsse
  BEGIN
    -- abk_dbrid, sollMenge, fertMenge
    SELECT abk.dbrid, COALESCE(ld_stk_soll_uf1, ld_stk_uf1), ld_stk_uf1 INTO abk_dbrid, sollMenge, fertMenge
    FROM abk LEFT JOIN ldsdok ON ld_id = ab_ld_id WHERE ab_ix = in_abix;

    IF abk_dbrid IS NULL THEN
        RETURN;
    END IF;

    -- Fertigungsmenge - der aktuellen Ausschüsse ist die maximale, effektive Fertigungsmenge
    effektiveMenge:=
        -- kleinste, insgesamt gemeldete Menge der geschlossenen AGs der ABK. Ansonsten geplante Fertigungsmenge. Mehr kann nicht produziert werden.
        COALESCE(nullif(tabk.get_bdea_fert(in_abix, true), 0), fertMenge)
        -- Minus
        -
        -- Summe der aktuellen Ausschüsse aller offenen AGs der ABK.
        tabk.get_bdea_ausschuss(in_abix, false);
    --

    -- effektive Fertigungsmenge (auch ggf. korrigierte Menge) unterschreitet (erneut) Sollmenge.
    IF effektiveMenge < sollMenge THEN
        IF COALESCE(TRecnoParam.GetNumeric('System.Freigabemenge.Unterproduktion', abk_dbrid), effektiveMenge-1) <> effektiveMenge THEN -- Freigabemenge ändert sich und muss damit erneut freigegeben werden
            PERFORM TRecnoParam.Set('System.Freigabemenge.Unterproduktion', abk_dbrid, effektiveMenge);
            PERFORM TRecnoParam.Set('System.Quittierung.Unterproduktion', abk_dbrid, false);
        END IF;
    ELSE -- gemeldete Fertigungsmenge wird noch oben korrigiert und unterschreitet Soll nicht (mehr)
        PERFORM TRecnoParam.Delete('System.Freigabemenge.Unterproduktion', abk_dbrid);
        PERFORM TRecnoParam.Delete('System.Quittierung.Unterproduktion', abk_dbrid);
    END IF;

    RETURN;
  END $$ LANGUAGE plpgsql VOLATILE STRICT;
--

-- Haupt ABK einer Baugruppe für Stückliste oder

CREATE OR REPLACE FUNCTION tabk.abk_main_abk( _abix integer ) RETURNS integer
  AS $$

    -- returns itself if no parent is present

     WITH RECURSIVE _tree AS (
      SELECT a.ab_ix, a.ab_parentabk, 0 AS depth FROM public.abk a WHERE ab_ix = _abix
      UNION
      SELECT a.ab_ix, a.ab_parentabk, depth - 1 FROM _tree p JOIN public.abk a ON a.ab_ix = p.ab_parentabk
    )

    SELECT ab_ix FROM _tree ORDER BY depth LIMIT 1;

  $$ LANGUAGE sql STABLE STRICT PARALLEL SAFE;

  CREATE OR REPLACE FUNCTION tplanterm.abk_main_abk(abix INTEGER) RETURNS INTEGER AS $$ SELECT tabk.abk_main_abk(abix); $$ LANGUAGE SQL;
--

--
CREATE OR REPLACE FUNCTION tabk.get_abk_kosten(
      _abix                       integer,              -- ABK
      _stufe                      integer,              -- Soll- und/oder Istdaten, s.o.
      _stichtag                   date = current_date,  -- zum Stichtag
      -- UE-Ergebnis (Stufe 1) wird mittels VKP-Faktor als VKP-Basis ausgegeben. Sonst ist Standard als Selbskosten.
      _ue_abk_kost_per_vkp_faktor boolean = TSystem.Settings__GetBool( 'UE.Bewertung.per.VKP-Faktor', false ),

      -- Arbeitszeit und -Kosten
      OUT arbeitszeit_soll      numeric,
      OUT arbeitszeit_ist       numeric,
      OUT arbeitszeit_kost_soll numeric,
      OUT arbeitszeit_kost_ist  numeric,

      -- Materialkosten
      OUT mat_kost_soll         numeric,
      OUT mat_kost_ist          numeric,
      OUT roh_mat_kost_soll     numeric,
      OUT roh_mat_kost_ist      numeric,
      OUT norm_mat_kost_soll    numeric,
      OUT norm_mat_kost_ist     numeric,
      OUT em_mat_kost_soll      numeric,
      OUT em_mat_kost_ist       numeric,

      -- Auswärtskosten
      OUT aw_kost_soll          numeric,
      OUT aw_kost_ist           numeric,

      -- Kosten durch globale Zuschläge
      OUT abk_zko_kost_soll     numeric,
      OUT abk_zko_kost_ist      numeric,

      -- Sonderkosten
      -- kein Soll vorhanden
      OUT sond_kost_ist         numeric,

      -- Summen der ABK-Kosten
      -- Herstellkosten laut Zuschlagskalkulation
      OUT abk_hk_soll           numeric,
      OUT abk_hk_ist            numeric,

      -- Selbstkosten laut Zuschlagskalkulation
      OUT abk_sk_soll           numeric,
      OUT abk_sk_ist            numeric,

      -- Kalkulationsergebnis entspr. Stufe
      OUT abk_kost_soll         numeric,
      OUT abk_kost_ist          numeric,

      -- Stückkosten aufgrund Kalkulationsergebnis
      OUT abk_stk_kost_soll     numeric,
      OUT abk_stk_kost_ist      numeric

    ) RETURNS record AS $$
  DECLARE
      _abk_zko_proz         numeric;
      _vkp_faktor           numeric;
      _fert_menge           record;
      _menge_abk_kost_ist   numeric;
  BEGIN
    -- Hauptfunktion für alle Teil- und Summenergebnisse der ABK-Kosten für Soll und Ist nach Stichtag
      -- Stufe 0: nur Solldaten (Vorkalkulation)
      -- Stufe 1: Soll- und Istdaten ohne NK (UE)
      -- Stufe 2: Soll- und Istdaten inkl. NK (Allgemein)
      -- Stufe 3: nur Istdaten (Nach- und Zwischenkalkulation)
      -- Details, siehe Felder und Beschreibung der Unterfunktionen
      -- Gemeinkosten sind in Unterfunktionen für Soll und Ist, laut Zuschlagskalkulation #12933.
      -- Globale Zuschläge (ZKO) laut Zuschlagskalkulation #12933 und #15249
      -- UE-Ergebnis (Stufe 1) kann als VKP-Basis oder als Selbskosten ausgegeben werden, siehe #15479.
      -- VKP-Basis entspr. Fertigungsartikel, siehe #15067.
      -- Default ist Globales Setting bzw. Selbstkosten. Kann in UE-RTFs optional überdeckt werden.


    -- Return NULL, wenn ABK nicht existiert.
    IF NOT EXISTS(SELECT true FROM abk WHERE ab_ix = _abix) THEN  RETURN;   END IF;


    -- 1. Arbeitszeit und -Kosten
        SELECT ak.arbeitszeit_soll, ak.arbeitszeit_ist, ak.arbeitszeit_kost_soll, ak.arbeitszeit_kost_ist
        INTO      arbeitszeit_soll,    arbeitszeit_ist,    arbeitszeit_kost_soll,    arbeitszeit_kost_ist
        FROM tabk.get_arbeitszeit_kosten( _abix, _stufe, _stichtag ) AS ak
        ;
    --


    -- 2. Materialkosten
        SELECT mk.mat_kost_soll,      mk.mat_kost_ist,
               mk.roh_mat_kost_soll,  mk.roh_mat_kost_ist,
               mk.norm_mat_kost_soll, mk.norm_mat_kost_ist,
               mk.em_mat_kost_soll,   mk.em_mat_kost_ist
        INTO      mat_kost_soll,         mat_kost_ist,
                  roh_mat_kost_soll,     roh_mat_kost_ist,
                  norm_mat_kost_soll,    norm_mat_kost_ist,
                  em_mat_kost_soll,      em_mat_kost_ist
        FROM tabk.get_mat_kosten( _abix, _stufe, _stichtag ) AS mk
        ;
    --


    -- 3. Auswärtskosten
        SELECT awk.aw_kost_soll, awk.aw_kost_ist
        INTO       aw_kost_soll,     aw_kost_ist
        FROM tabk.get_aw_kosten( _abix, _stufe, _stichtag ) AS awk
        ;
    --


    -- Zwischensumme: Herstellkosten
        -- Fertigungskosten, Materialien/Unterbauteile, Auswärtskosten
        abk_hk_soll := arbeitszeit_kost_soll + mat_kost_soll + aw_kost_soll;
        abk_hk_ist  := arbeitszeit_kost_ist  + mat_kost_ist  + aw_kost_ist;
    --


    -- 4. Globale Zuschläge
        _abk_zko_proz :=
              -- Satz aller Zuschläge
              sum( n7zk_proz ) / 100
            FROM nk7zko
            WHERE n7zk_ab_ix = _abix
              AND n7zk_proz IS NOT NULL
        ;

        -- angewandt auf Wertschöpfungsebene
          -- also Herstellkosten ohne Unterbauteile, siehe #15249
        abk_zko_kost_soll := coalesce( _abk_zko_proz, 0 ) * ( arbeitszeit_kost_soll + roh_mat_kost_soll + norm_mat_kost_soll + aw_kost_soll );
        abk_zko_kost_ist  := coalesce( _abk_zko_proz, 0 ) * ( arbeitszeit_kost_ist  + roh_mat_kost_ist  + norm_mat_kost_ist  + aw_kost_ist  );
    --


    -- 5. Sonderkosten
        -- Entstehen erst nach Abschluss der Fertigung. Spielen lediglich in der Zwischenkalkulation (Ist) und Nachkalkulation eine Rolle.
        -- Trotzdem in UE einbeziehen, da Relevanz bei Kunden uneindeutig. Ist explizit ausgewiesen und kann ggf. rausgerechnet werden.
        IF _stufe >= 1 THEN
            SELECT sk.sond_kost_ist
            INTO      sond_kost_ist
            FROM tabk.get_sonder_kosten( _abix, _stichtag ) AS sk
            ;

        END IF;

        sond_kost_ist := coalesce( sond_kost_ist, 0 );
    --


    -- Zwischensumme: Selbstkosten
        -- Herstellkosten + Zuschläge + Sonderkosten
        abk_sk_soll := abk_hk_soll + abk_zko_kost_soll;
        abk_sk_ist  := abk_hk_ist  + abk_zko_kost_ist  + sond_kost_ist;
    --


    -- Kalkulationsergebnis
        -- UE ist VKP-Basis entspr. Fertigungsartikel
        IF _stufe = 1 AND _ue_abk_kost_per_vkp_faktor THEN

            _vkp_faktor :=
                  coalesce( ak_vkpfaktor, ac_vkpfaktor, 1 )
                FROM abk
                  LEFT JOIN ldsdok ON ld_abk = ab_ix
                  LEFT JOIN art ON ak_nr = coalesce( ld_aknr, ab_ap_nr )
                  LEFT JOIN artcod ON ac_n = ak_ac
                WHERE ab_ix = _abix
                -- Ersten erstellten PA bei mehrfacher Verknüpfung bevorzugen, siehe #15362.
                ORDER BY ld_id
                LIMIT 1
            ;

            abk_kost_soll := abk_sk_soll * _vkp_faktor;
            abk_kost_ist  := abk_sk_ist  * _vkp_faktor;

        -- sonst Selbstkosten
        ELSE
            abk_kost_soll := abk_sk_soll;
            abk_kost_ist  := abk_sk_ist;
        END IF;
    --


    -- Berechnung der Stückkosten
        -- Fertigungsmengen für Stückkosten-Berechnung
        SELECT
          -- stk:       geplante gesamte Fertigungsmenge (inkl. geplantem Ausschuss)
          coalesce( ld_stk_uf1, ab_st_uf1 ) AS stk,
          -- stk_soll:  geplante Soll-Fertigungsmenge (nur Gutteile)
          coalesce( ld_stk_soll_uf1, ld_stk_uf1, ab_st_uf1_soll, ab_st_uf1 ) AS stk_soll,
          -- stkl:      gebuchte Fertigungsmenge
          ld_stkl AS stkl,
          -- auss:      gebuchter Ausschuss
          0::numeric AS auss
        FROM abk
          LEFT JOIN ldsdok ON ld_abk = ab_ix
        WHERE ab_ix = _abix
        INTO _fert_menge
        ;

        -- Ausschuss berücksichtigen
        -- IF _stufe >= 2 THEN
        --    _fert_menge.auss := coalesce( abk.get_bdea_ausschuss( _abix ), 0 );
        -- END IF;

        -- Stückkosten für Soll
        IF _fert_menge.stk_soll > 0 THEN
            -- Soll-Kosten / Fertigungsmenge Soll
            abk_stk_kost_soll := abk_kost_soll / _fert_menge.stk_soll;
        END IF;

        -- aktuelle Fertigungsmenge Soll für ABK-Ist-Kosten als Berechnungsgrundlage für Stückkosten Ist
        -- gebuchte bzw. geplante Fertigungsmenge
        _menge_abk_kost_ist :=
            -- greatest: gebuchte Fertigungsmenge ist tatsächlich höher als geplante Fertigungsmenge
            greatest(
                _fert_menge.stkl,
                -- least: Fertigungsmenge (Ausschuss berücksichtigt) unterschreitet Sollmenge. Damit max. geplante Fertigungsmenge.
                least(
                    _fert_menge.stk - _fert_menge.auss,
                    _fert_menge.stk_soll
                )
            )
        ;

        -- Stückkosten für Ist
        IF _menge_abk_kost_ist > 0 THEN
            -- Ist-Kosten / aktuelle Fertigungsmenge Soll
            abk_stk_kost_ist := abk_kost_ist / _menge_abk_kost_ist;
        END IF;
    --


    RETURN;
  END $$ LANGUAGE plpgsql STABLE STRICT;
--

-- Artikelnummer in ABK ändern
CREATE OR REPLACE FUNCTION tabk.abk_aknr_change__auftg_ldsdok(_aknr VARCHAR, _abix INTEGER, _old_akbez VARCHAR) RETURNS VOID AS $$
  DECLARE akbez VARCHAR;
          ldmce INTEGER;
  BEGIN
    -- Berechtigung prüfen
    IF  EXISTS( SELECT true FROM pg_group WHERE (groname='SYS.AVOR' OR groname='LOLL.AV') ) THEN
        IF NOT EXISTS( SELECT true FROM pg_group JOIN pg_user ON usesysid=ANY(grolist) WHERE usename=current_user AND (groname='SYS.AVOR' OR groname='LOLL.AV') ) THEN
            RAISE EXCEPTION '% (Group SYS.AVOR)', lang_text(6338);
        END IF;
    END IF;

    -- Abbruch, wenn Artikelnummer nicht existiert
    IF NOT EXISTS(SELECT true FROM art WHERE ak_nr=_aknr) THEN
        RAISE EXCEPTION '% : %', lang_text(20105), _aknr;
    END IF;

    -- Abbruch, wenn neuer Artikel nicht die passende ME besitzt, die in Bestellung gesetzt ist
    IF EXISTS(SELECT TRUE FROM ldsdok WHERE ld_abk=_abix)
        AND NOT EXISTS( SELECT true FROM artmgc WHERE m_ak_nr=_aknr AND m_mgcode=(SELECT m_mgcode FROM artmgc JOIN ldsdok ON m_id=ld_mce WHERE ld_abk=_abix) )
    THEN
        RAISE EXCEPTION '%', lang_text(20106);
    END IF;

    -- Abbruch, wenn bereits Wareneingang zu ABK existiert (verbucht)
    IF EXISTS(SELECT true FROM wendat JOIN ldsdok ON w_lds_id=ld_id WHERE ld_abk=_abix) THEN
        RAISE EXCEPTION '%', lang_text(20108);
    END IF;

    -- Abbruch, wenn bereits Lagerabgang bzw. Lieferschein zu ABK existiert
    IF EXISTS(SELECT true FROM lifsch JOIN auftg ON l_ag_id=ag_id WHERE ag_astat='I' AND ag_ownabk=_abix)
        OR EXISTS(SELECT true FROM belegpos WHERE belp_ab_ix=_abix)
    THEN
        RAISE EXCEPTION '%', lang_text(20107);
    END IF;

    -- Abbruch, wenn keine Standard-ASK beim neuen Artikel vorhanden ist.
    IF NOT EXISTS(SELECT true FROM opl WHERE op_n=_aknr AND op_standard) THEN
        RAISE EXCEPTION '%', lang_text(16478);
    END IF;

    -- neue Artikelbezeichnung
    akbez:=ak_bez FROM art WHERE ak_nr=_aknr;

    -- interne Aufträge umschreiben, in denen ich selbst als ABK direkt eingehe
    UPDATE auftg SET ag_aknr=_aknr WHERE ag_astat='I' AND ag_ownabk=_abix;

    IF COALESCE(_old_akbez, '') <> '' THEN
        UPDATE auftg SET ag_akbz=akbez WHERE ag_astat='I' AND ag_ownabk=_abix AND ag_aknr=_aknr AND ag_akbz=_old_akbez;
    END IF;

    ldmce:=ld_mce FROM ldsdok WHERE ld_abk=_abix;

    -- interne Bestellung umschreiben
    UPDATE ldsdok SET ld_aknr=_aknr WHERE ld_abk=_abix;

    -- Mengeneinheitszuordnung aktualisieren
    UPDATE ldsdok SET ld_mce=(SELECT m_id FROM artmgc WHERE m_ak_nr=_aknr AND m_mgcode=(SELECT m_mgcode FROM artmgc WHERE m_id=ldmce)) WHERE ld_abk=_abix;

    IF COALESCE(_old_akbez, '') <> '' THEN
       UPDATE ldsdok SET ld_akbz=akbez WHERE ld_abk=_abix AND ld_aknr=_aknr AND ld_akbz=_old_akbez;
    END IF;

    -- ABK selbst umschreiben
    UPDATE abk SET ab_askix=(SELECT op_ix FROM opl WHERE op_n=_aknr AND op_standard) WHERE ab_ix=_abix;
    --... nur wenn vorher schon ab_ap_nr gesetzt war
    UPDATE abk SET ab_ap_nr=_aknr WHERE ab_ap_nr IS NOT NULL AND ab_ix=_abix;

    PERFORM tartikel.bestand_abgleich(_aknr);

  END $$ LANGUAGE plpgsql;
--

-- Arbeitszeit, Kosten pro Kostenstelle zu einer ABK
CREATE OR REPLACE FUNCTION tabk.get_arbeitszeit_nk_pro_ks(
        in_abix  integer,
  INOUT in_ag    integer DEFAULT 0,
        in_dat   date    DEFAULT current_date,
        _ba_minr integer DEFAULT null, -- https://redmine.prodat-sql.de/issues/20319 optionaler MA, auf den die Berechnung eingeschränkt wird
    OUT zeit     numeric,
    OUT kosten   numeric,
    OUT ks       varchar(9)
)
RETURNS SETOF record AS $$
  DECLARE r record;
          bdea_daten record;
          rm_daten record;
          nk2_daten record;
          _zeit_alle numeric;
          _nk_faktor numeric;
  BEGIN
    FOR r in SELECT ba_ks AS ks
             FROM bdea
             WHERE ba_ix = in_abix
                AND ba_anf::date <= in_dat
                AND NOT ba_in_rckmeld
                AND IFTHEN( in_ag = 0, ba_op > 0, ba_op = in_ag )
             UNION
             SELECT r_ks AS ks
             FROM ab2 JOIN rm ON r_a2_id = a2_id
             WHERE a2_ab_ix = in_abix
                AND r_da::date <= in_dat
                AND NOT a2_buch
                AND IFTHEN( in_ag = 0, a2_n > 0, a2_n = in_ag )
             UNION
             SELECT n2_ks AS ks
             FROM nk2
             WHERE n2_ix = in_abix
                AND IFTHEN( in_ag = 0, n2_n > 0, n2_n = in_ag )
      LOOP
            -- Auftragszeiten und -kosten nicht in Rückmeldungen
            -- ggf. nur auf einen Mitarbeiter beschränkt (#20319)
            SELECT
              sum( ba_efftime ) FILTER ( WHERE IFTHEN( _ba_minr IS null, true, ba_minr = _ba_minr )) AS zeit,
              sum( ba_efftime ) AS zeit_alle,
              sum( ba_efftime * ks_sts ) FILTER ( WHERE IFTHEN( _ba_minr IS null, true, ba_minr = _ba_minr )) AS kosten,
              sum( ba_efftime * ks_sts ) AS kosten_alle INTO bdea_daten
            FROM bdea
              JOIN tartikel.ksv__data__by__table__get( bdea ) ON true
            WHERE ba_ix = in_abix
              AND IFTHEN( in_ag = 0, ba_op > 0, ba_op = in_ag )
              AND ba_anf::date <= in_dat
              AND ba_ks = r.ks;

            -- Rückmeldungszeiten und -kosten nicht verbucht
            -- ggf. nur auf einen Mitarbeiter beschränkt (#20375)
            SELECT
              sum( r_std_sek / 3600 ) FILTER ( WHERE IFTHEN( _ba_minr IS null, true, r_minr = _ba_minr )) AS zeit,
              sum( r_std_sek / 3600 ) AS zeit_alle,
              sum( r_std_sek * ks_sts / 3600 ) FILTER ( WHERE IFTHEN( _ba_minr IS null, true, r_minr = _ba_minr )) AS kosten,
              sum( r_std_sek / 3600 * ks_sts ) AS kosten_alle INTO rm_daten
            FROM ab2
              JOIN rm ON r_a2_id = a2_id
              JOIN tartikel.ksv__data__by__table__get( rm ) ON true
            WHERE a2_ab_ix = in_abix
              AND IFTHEN( in_ag = 0, a2_n > 0, a2_n = in_ag )
              AND r_da::date <= in_dat
              AND r_ks = r.ks;

            -- Nachkalkulationszeiten und -kosten (verbuchte)
            -- #20375 Die Nachkalkulationszeiten und -kosten werden anteilig zu den Stempelzeiten auf die MA verteilt.
            IF _ba_minr IS NOT null THEN
              _zeit_alle := coalesce( bdea_daten.zeit_alle, 0 );
              IF _zeit_alle = 0 THEN
                _zeit_alle := 1;
              END IF;
              _nk_faktor := coalesce( bdea_daten.zeit, 0) / _zeit_alle;
            ELSE
              _nk_faktor := 1;
            END IF;

            SELECT
              sum( n2_ez_stu ) * _nk_faktor AS zeit,
              sum( n2_ez_stu *
                CASE WHEN n2_ruest   THEN ks_stsr -- Rüststundensatz
                     WHEN n2_pz_para THEN ks_stsm -- Stundensatz für parallele Personalzeit
                     ELSE ks_sts                  -- Stundensatz
                END ) * _nk_faktor AS kosten
            INTO nk2_daten
            FROM nk2
              LEFT JOIN tartikel.ksv__data__by__table__get( nk2 ) ON true
            WHERE n2_ix = in_abix
              AND IFTHEN( in_ag = 0, n2_n > 0, n2_n = in_ag )
              AND n2_ks = r.ks;

            zeit := coalesce( nk2_daten.zeit, rm_daten.zeit, bdea_daten.zeit, 0 );
            kosten := coalesce( nk2_daten.kosten, rm_daten.kosten, bdea_daten.kosten, 0 );

            ks := r.ks;

            RETURN NEXT;
      END LOOP;
  END $$ LANGUAGE plpgsql STABLE;
--

-- #20407 aggregiert die Stempeleungen eines Mitarbeiters zu einem Arbeitsgang und ermittelt
-- gefertigte Stückzahl, Ausschuss, Arbeitsplätze, Zeit, Kosten und Kostenstelle
CREATE OR REPLACE FUNCTION tabk.get__bdea__pro__ks__minr(
        _abix      integer,
  INOUT ag_n       integer,
        _ba_minr   integer, -- https://redmine.prodat-sql.de/issues/20319 optionaler MA, auf den die Berechnung eingeschränkt wird
    OUT stueck     numeric,
    OUT ausschuss  numeric,
    OUT plaetze    varchar,
    OUT zeit       numeric,
    OUT kosten     numeric,
    OUT ks         varchar
)
RETURNS record AS $$
BEGIN

  SELECT
         sum( ba_stk ), sum( ba_auss ), string_agg( DISTINCT ba_ksap, ',' ), min( az.zeit ), min( az.kosten ), string_agg( DISTINCT az.ks, ',' )
    INTO        stueck,      ausschuss,                             plaetze,           zeit,           kosten,                                ks
  FROM bdea
  CROSS JOIN tabk.get_arbeitszeit_nk_pro_ks( _abix, ag_n, current_date, _ba_minr ) AS az
  WHERE
         ba_ix = _abix
     AND ba_op = ag_n
     AND ba_minr = _ba_minr
  GROUP BY ba_minr
  ORDER BY ba_minr;

  stueck := coalesce( stueck, 0 );
  ausschuss := coalesce( ausschuss, 0 );
  zeit := coalesce( zeit, 0 );
  kosten := coalesce( kosten, 0 );

END $$ LANGUAGE plpgsql STABLE;
--

--
CREATE OR REPLACE FUNCTION tabk.get_arbeitszeit_kosten(
    _abix                       integer,              -- ABK
    _stufe                      integer,              -- Soll- und/oder Istdaten, s.u.
    _stichtag                   date = current_date,  -- zum Stichtag
    OUT arbeitszeit_soll        numeric, -- Arbeitszeit Soll
    OUT arbeitszeit_ist         numeric, -- Arbeitszeit Ist
    OUT arbeitszeit_kost_soll   numeric, -- Arbeitszeit Sollkosten
    OUT arbeitszeit_kost_ist    numeric  -- Arbeitszeit Istkosten
    ) RETURNS record AS $$
    DECLARE
        _abk_soll   record;
        _abk_ist    record;
        _nk_ist     record;
        _ta_pz_para numeric;
    BEGIN
      -- Arbeitskosten der ABK für Soll und Ist nach Stichtag
        -- Stufe 0: nur Solldaten laut ABK (Vorkalkulation)
        -- Stufe 1: Soll- und Istdaten der ABK ohne NK (UE)
        -- Stufe 2: Soll- und Istdaten der ABK inkl. NK (Allgemein)
        -- Stufe 3: nur Istdaten (Nach- und Zwischenkalkulation)
        -- Beachte: ABK-NK-Stundensätze und ABK-NK-Gemeinkostenzuschläge (nach ABK-Erstellung verfügbar) gelten für alle Stufen.
  
  
      -- Return NULL, wenn ABK nicht existiert.
      IF NOT EXISTS(SELECT true FROM abk WHERE ab_ix = _abix) THEN  RETURN;   END IF;
  
  
      -- Solldaten laut ABK
      -- Gemäß Vorkalkulation aus ASK
      -- Stufen 0,1,2
      IF _stufe <= 2 THEN
          -- Arbeitszeit und Arbeitszeit-Kosten anhand der ABK-AG ermitteln.
          SELECT
            -- gesamte Auftragszeit T
            sum( a2_ta ) AS zeit,
  
            -- Kosten für Rüsten (tr)
            sum(
                -- Rüstzeit in h
                  coalesce( a2_tr_sek / 3600, 0 )
                -- Normstundensatz Rüsten
                * coalesce( nkk_nssr, ks_stsr ) -- ABK-NK-Stundensatz bevorzugen
                -- Rüstkostenzuschlag
                * ( 1 + ab_nk_rgk / 100 )
            ) AS tr_kosten,
  
            -- Kosten für Ausführung (ta)
            sum(
                -- Ausführungszeit (Hauptzeit + Nebenzeit + Verteilzeit + Mitarbeiterzeit) in h
                  ( a2_ta - coalesce( a2_tr_sek / 3600, 0 ) ) -- implizit entspr. Berechnung ta (alles andere als tr)
                -- Normstundensatz Fertigen (Ausführung)
                * coalesce( nkk_nssf, ks_sts ) -- ABK-NK-Stundensatz bevorzugen
                -- Fertigungsgemeinkosten
                * ( 1 + ab_nk_fgk / 100 )
            ) AS ta_kosten
  
          FROM ab2
            JOIN abk ON ab_ix = a2_ab_ix
            JOIN ksv ON ks_abt = a2_ks
            LEFT JOIN nkksv ON nkk_ab_ix = a2_ab_ix AND nkk_ks = a2_ks
          WHERE a2_ab_ix = _abix
            AND NOT a2_ausw
  
          INTO _abk_soll
          ;
  
          -- Auftragszeit T
          arbeitszeit_soll      := coalesce( _abk_soll.zeit, 0 );
          -- Auftragszeit-Kosten für T: Rüstkosten für tr + Ausführungskosten für ta
          arbeitszeit_kost_soll := coalesce( _abk_soll.tr_kosten, 0 ) + coalesce( _abk_soll.ta_kosten, 0 );
  
      END IF;
  
  
      -- Istdaten der ABK ohne NK
      -- Gemäß Auftragszeiten und Rückmeldungen
      -- Zustand: AG offen und nicht verbucht und AG geschlossen und nicht verbucht
      -- Stufen 1,2,3
      IF _stufe >= 1 THEN
          -- Arbeitszeit und Arbeitszeit-Kosten anhand der gestempelten Auftragszeiten und Rückmeldungen ermitteln.
  
          SELECT
            -- gesamte Auftragszeit T
            sum ( bdea_rm_daten.zeit ) AS zeit,
  
            -- Kosten für Rüsten (tr)
            sum(
                -- Rüstzeit in h
                  bdea_rm_daten.zeit
                -- Normstundensatz Rüsten
                * coalesce( nkk_nssr, ks_stsr ) -- ABK-NK-Stundensatz bevorzugen
                -- Rüstkostenzuschlag
                * ( 1 + ab_nk_rgk / 100 )
            )
              -- Rüsten
              FILTER ( WHERE bdea_rm_daten.ruesten )
            AS tr_kosten,
  
            -- Kosten für Ausführung (ta)
            sum(
                -- Ausführungszeit (Hauptzeit + Nebenzeit + Verteilzeit + Mitarbeiterzeit) in h
                  bdea_rm_daten.zeit
                -- Normstundensatz Fertigen (Ausführung)
                * coalesce( nkk_nssf, ks_sts ) -- ABK-NK-Stundensatz bevorzugen
                -- Fertigungsgemeinkosten
                * ( 1 + ab_nk_fgk / 100 )
            )
              -- Ausführung
              FILTER ( WHERE NOT bdea_rm_daten.ruesten )
            AS ta_kosten
  
          FROM
            -- ABK-Ist: Auftragszeiten und Rückmeldungen (bdea_rm_daten)
            (
                -- 1. Auftragszeiten nicht in Rückmeldungen
                SELECT
                  ba_ix                   AS abix,
                  ba_ks                   AS ks,
                  ba_ruest                AS ruesten,
                  -- Summe der Zeit pro Kostenstelle und Rüsten/Ausführung in h
                  sum( ba_efftime )       AS zeit
  
                -- Auftragszeiten
                FROM bdea
                WHERE ba_ix = _abix
                  -- nach Stichtag
                  AND ba_anf::date <= _stichtag
                  -- nicht in Rückmeldung überführt. Dafür ist rm zuständig.
                  AND NOT ba_in_rckmeld
                GROUP BY ba_ix, ba_ks, ba_ruest
  
                -- 2. Zeiten der Rückmeldungen
                UNION ALL
                SELECT
                  a2_ab_ix                AS abix,
                  r_ks                    AS ks,
                  r_ruest                 AS ruesten,
                  -- Summe der Zeit pro Kostenstelle und Rüsten/Ausführung in h
                  sum( r_std_sek ) / 3600 AS zeit
  
                -- Rückmeldungen an ABK-AG
                FROM ab2
                  JOIN rm ON r_a2_id = a2_id
                WHERE a2_ab_ix = _abix
                  -- nach Stichtag
                  AND r_da::date <= _stichtag
                  -- Verbuchungsstatus entspr. Ausführungsstufe (UE) berücksichtigen
                  AND CASE
                          -- Bei UE auch verbuchte einbeziehen, da nach Stichtag.
                          WHEN _stufe = 1 THEN true
                          -- sonst nur nicht verbucht durch NK, da sonst NK-Daten gelten.
                          ELSE NOT a2_buch
                      END
                GROUP BY a2_ab_ix, r_ks, r_ruest
  
          ) AS bdea_rm_daten
            JOIN abk ON ab_ix   = bdea_rm_daten.abix
            JOIN ksv ON ks_abt  = bdea_rm_daten.ks
            LEFT JOIN nkksv ON nkk_ab_ix = bdea_rm_daten.abix AND nkk_ks = bdea_rm_daten.ks
  
          INTO _abk_ist
          ;
  
          -- Auftragszeit T
          arbeitszeit_ist       := coalesce( _abk_ist.zeit, 0 );
          -- Kosten für Auftragszeit (T): Kosten für Rüsten (tr) + Kosten für Ausführung (ta)
          arbeitszeit_kost_ist  := coalesce( _abk_ist.tr_kosten, 0 ) + coalesce( _abk_ist.ta_kosten, 0 );
  
      END IF;
  
  
      -- Istdaten der ABK inkl. NK (kein Stichtag mgl.)
      -- Gemäß übertragener Daten in Nachkalkulation
      -- Zustand: AG geschlossen und verbucht.
      -- Beachte: unverbuchte Zeiten (s.o.) werden mit einbezogen (fließender Übergang).
      -- Stufen 2,3
      IF _stufe >= 2 THEN
          -- Nachkalkulationszeiten und -kosten (verbuchte)
            -- vgl. tplanterm.buchrm
          _nk_ist := tplanterm.nk__kosten_fertigung_tr_ta__sum( _abix );
  
          -- Kosten für die parallelen Personalkosten
          _ta_pz_para := tplanterm.nk__pz_para__sum( _abix );
  
          -- Auftragszeit T
          -- Einbeziehung der unverbuchten Daten (s.o.).
          arbeitszeit_ist       :=  coalesce( arbeitszeit_ist, 0 )      + coalesce( _nk_ist.zeit, 0 );
          -- Kosten für Auftragszeit (T): Kosten für Rüsten (tr) + Kosten für Ausführung (ta)
          arbeitszeit_kost_ist  :=
              coalesce( arbeitszeit_kost_ist, 0 )
            + coalesce( _nk_ist.kosten_ruesten, 0 )
            + coalesce( _nk_ist.kosten_ausfuehrung, 0 )
            + _ta_pz_para;
      END IF;
  
      -- wenigstens 0 ausgeben
      arbeitszeit_soll      := coalesce( arbeitszeit_soll,      0 );
      arbeitszeit_kost_soll := coalesce( arbeitszeit_kost_soll, 0 );
      arbeitszeit_ist       := coalesce( arbeitszeit_ist,       0 );
      arbeitszeit_kost_ist  := coalesce( arbeitszeit_kost_ist,  0 );
  
  
      RETURN;
    END $$ LANGUAGE plpgsql STABLE STRICT;
--

--
CREATE OR REPLACE FUNCTION tabk.get_mat_kosten(
    _abix                   integer,              -- ABK
    _stufe                  integer,              -- Soll- und/oder Istdaten, s.o.
    _stichtag               date = current_date,  -- zum Stichtag
    OUT mat_kost_soll       numeric,  -- Material    Sollkosten gesamt (Roh + Norm + ggf. Unterbauteile)
    OUT mat_kost_ist        numeric,  -- Material    Istkosten  gesamt (Roh + Norm + ggf. Unterbauteile)
    OUT roh_mat_kost_soll   numeric,  -- Rohmaterial Sollkosten
    OUT roh_mat_kost_ist    numeric,  -- Rohmaterial Istkosten
    OUT norm_mat_kost_soll  numeric,  -- Normteile   Sollkosten
    OUT norm_mat_kost_ist   numeric,  -- Normteile   Istkosten
    OUT em_mat_kost_soll    numeric,  -- Einzel- und Montageteile (Unterbauteile, eigene Fertigung)  Sollkosten
    OUT em_mat_kost_ist     numeric   -- Einzel- und Montageteile (Unterbauteile, eigene Fertigung)  Istkosten
    ) RETURNS record AS $$
    DECLARE
        _per_stichtag     boolean;  -- Ist die aktuelle Berechnung per Stichtag (ungleich heute). Optimierung per Konstante.
    BEGIN
      -- Materialkosten der ABK für Soll und Ist nach Stichtag
        -- Stufe 0: nur Solldaten (Vorkalkulation)
        -- Stufe 1: Soll- und Istdaten ohne NK (UE)
        -- Stufe 2: Soll- und Istdaten inkl. NK (Allgemein)
        -- Stufe 3: nur Istdaten (Nach- und Zwischenkalkulation)
        -- Beachte: ABK-NK-Gemeinkostenzuschläge (nach ABK-Erstellung verfügbar) gelten für alle Stufen.
  
  
      -- Return NULL, wenn ABK nicht existiert.
      IF NOT EXISTS(SELECT true FROM abk WHERE ab_ix = _abix) THEN  RETURN;   END IF;
  
  
      -- Solldaten laut ABK-Materialpositionen
      -- Gemäß Vorkalkulation aus ASK bzw. Artikelstamm
      -- Stufen 0,1,2
      IF _stufe <= 2 THEN
          SELECT
            -- Rohmaterialien
            -- inkl. Materialgemeinkosten
            sum(
                  mat_kost_soll_pro_pos
                * abk_mgk
            )
              -- Rohmaterial ohne eigenen ABK
              -- analog tplanterm.buchrm
              FILTER ( WHERE ag_post2 = 'R' AND ag_ownabk IS NULL )
            AS kosten_rohmaterialien_soll,
  
  
            -- Normteile
            -- inkl. VKP-Faktor und ohne Materialgemeinkosten, vgl. #15331
            -- #19308 Per dynamischer Einstellung soll der Verkaufspreisfaktor für Normteile ignoriert werden können.
            sum(
                  mat_kost_soll_pro_pos
                * CASE WHEN tsystem.settings__getbool( 'PREIS_NORMTEILE_OHNE_VKP_FAKTOR' ) THEN 1 ELSE vkp_faktor END
            )
              -- Normteile ohne eigenen ABK
              -- analog tplanterm.buchrm
              FILTER ( WHERE ag_post2 = 'N' AND ag_ownabk IS NULL )
            AS kosten_normteile_soll,
  
  
            -- Einzel- und Montageteile
            -- ohne VKP-Faktor und ohne Materialgemeinkosten, vgl. #15068
            sum( mat_kost_soll_pro_pos )
              -- Einzel- und Montageteile und MatPos mit eigener ABK
              -- analog tplanterm.buchrm
              FILTER ( WHERE ( ag_post2 IN ('E', 'M') OR ag_ownabk IS NOT NULL ) )
            AS kosten_einzel_montageteile_soll
  
          FROM (
              SELECT
                -- Kriterien für Summenbildung
                ag_post2,   -- Material-Typ
                ag_ownabk,  -- eigene ABK
  
                -- Materialkosten-Soll pro Position
                (
                    -- geplante Menge
                      ag_stk_uf1
                    -- Selbstkosten (inkl. definierten Fallbacks)
                    * CASE
                          -- Wert aus VK bei ABK-Erstellung
                          WHEN ag_vkp_uf1 > 0 THEN ag_vkp_uf1
                          -- Selbstkosten laut Artikelstamm
                          WHEN ak_hest    > 0 THEN ak_hest
                          -- Verkaufspreis-Basis auf Selbstkosten normiert, vgl. stvtrs__b_i__calcprices
                          ELSE ak_vkpbas / coalesce( nullif( ak_vkpfaktor, 0 ), nullif( ac_vkpfaktor, 0 ), 1 )
                      END
                ) AS mat_kost_soll_pro_pos,
  
                -- Materialgemeinkosten laut ABK
                ( 1 + ab_nk_mgk / 100 ) AS abk_mgk,
  
                -- Verkaufspreis-Faktor laut Artikelstamm bzw. AC (für Normteil)
                coalesce( nullif( ak_vkpfaktor, 0 ), nullif( ac_vkpfaktor, 0 ), 1 ) AS vkp_faktor
  
              FROM auftg
                JOIN abk ON ab_ix = ag_parentabk
                JOIN art ON ak_nr = ag_aknr
                JOIN artcod ON ac_n = ak_ac
              WHERE ag_parentabk IN ( SELECT tplanterm.get_all_child_abk( _abix => _abix, _OnlySetAbk => true ) )
                AND ag_post2 IN ('R', 'N', 'E', 'M')
                AND NOT TSystem.Enum_GetValue( ag_stat, 'BK' ) -- keine Beistellung durch Kunden, siehe #11316
          ) AS mat_pos
  
          INTO
            roh_mat_kost_soll,
            norm_mat_kost_soll,
            em_mat_kost_soll
          ;
  
      END IF;
  
  
      -- Istdaten der ABK
      -- Gemäß Materialbuchungen
      -- Stufen 1,2,3
      IF _stufe >= 1 THEN
          _per_stichtag := coalesce( _stichtag <> current_date, false );  -- mit Konstante für JOIN LATERAL schneller
  
          SELECT
            -- Rohmaterialien
            -- inkl. Materialgemeinkosten
            sum(
                  mat_kost_ist_pro_pos
                * abk_mgk
            )
              -- Rohmaterial ohne eigenen ABK
              -- analog tplanterm.buchrm
              FILTER ( WHERE ag_post2 = 'R' AND ag_ownabk IS NULL )
            AS kosten_rohmaterialien_ist,
  
  
            -- Normteile
            -- inkl. VKP-Faktor und ohne Materialgemeinkosten, vgl. #15331
            -- #19308 Per dynamischer Einstellung soll der Verkaufspreisfaktor für Normteile ignoriert werden können.
            sum(
                  mat_kost_ist_pro_pos
                * CASE WHEN tsystem.settings__getbool( 'PREIS_NORMTEILE_OHNE_VKP_FAKTOR' ) THEN 1 ELSE vkp_faktor END
            )
              -- Normteile ohne eigenen ABK
              -- analog tplanterm.buchrm
              FILTER ( WHERE ag_post2 = 'N' AND ag_ownabk IS NULL )
            AS kosten_normteile_ist,
  
  
            -- Einzel- und Montageteile
            -- ohne VKP-Faktor und ohne Materialgemeinkosten, vgl. #15068
            sum( mat_kost_ist_pro_pos )
              -- Einzel- und Montageteile und MatPos mit eigener ABK
              -- analog tplanterm.buchrm
              FILTER ( WHERE ( ag_post2 IN ('E', 'M') OR ag_ownabk IS NOT NULL ) )
            AS kosten_einzel_montageteile_ist
  
          FROM (
              SELECT
                -- Kriterien für Summenbildung
                ag_post2,   -- Material-Typ
                ag_ownabk,  -- eigene ABK
  
                -- Materialkosten-Ist pro Position
                (
                    -- ohne NK-Werte
                    CASE WHEN _stufe = 1 THEN
  
                        -- verbuchte Menge nach Stichtag
                          coalesce( ag_stkl_fromlifsch_todate, ag_stkl )
                        -- Selbstkosten
                          -- reale Selbstkosten des direkt nachkalkulierten Unterbauteils, vor NK: analog tplanterm.auftg_nk_calc_vkp
                          -- bzw. laut Selbstkosten (VK, Einkaufspreis) zum Zeitpunkt der ABK-Erstellung
                        * coalesce( nullif( own_abk.ab_nk_et, 0 ), ag_vkp_uf1 )
  
                    -- inkl. NK-Werte
                    ELSE
  
                        -- verbuchte Menge, inkl. Korrektur laut NK
                          coalesce( ag_nk_stkl_uf1, ag_stkl_fromlifsch_todate, ag_stkl )
                        -- Selbstkosten
                          -- Korrektur laut NK, NK-Durchschnittspreis laut Verbuchung, Preis laut VK bzw. Einkauf bei ABK-Erstellung
                          -- Beachte: nachkalkulierte Unterbauteile sind in NK-Durchschnittspreis enthalten.
                        * coalesce( ag_nk_vkp_uf1, ag_nk_calc_vkp_uf1, ag_vkp_uf1 )
  
                    END
                ) AS mat_kost_ist_pro_pos,
  
                -- Materialgemeinkosten laut ABK
                ( 1 + abk.ab_nk_mgk / 100 ) AS abk_mgk,
  
                -- Verkaufspreis-Faktor laut Artikelstamm bzw. AC (für Normteil)
                coalesce( nullif( ak_vkpfaktor, 0 ), nullif( ac_vkpfaktor, 0 ), 1 ) AS vkp_faktor
  
              FROM auftg
                JOIN abk ON ab_ix = ag_parentabk
                JOIN art ON ak_nr = ag_aknr
                JOIN artcod ON ac_n = ak_ac
                -- Direktes Unterbauteil, welches verbucht und nachkalkuliert ist.
                LEFT JOIN abk AS own_abk ON own_abk.ab_ix = ag_ownabk AND own_abk.ab_buch
                -- nach Stichtag
                LEFT JOIN LATERAL tauftg.ag_stkl_fromlifsch_todate( _stichtag, ag_id ) ON _per_stichtag
              WHERE ag_parentabk IN ( SELECT tplanterm.get_all_child_abk( _abix => _abix, _OnlySetAbk => true ) )
                AND ag_post2 IN ('R', 'N', 'E', 'M')
                AND NOT TSystem.Enum_GetValue( ag_stat, 'BK' ) -- keine Beistellung durch Kunden, siehe #11316
          ) AS mat_pos
  
          INTO
            roh_mat_kost_ist,
            norm_mat_kost_ist,
            em_mat_kost_ist
          ;
  
      END IF;
  
      -- wenigstens 0 ausgeben
      roh_mat_kost_soll   := coalesce( roh_mat_kost_soll,  0 );
      norm_mat_kost_soll  := coalesce( norm_mat_kost_soll, 0 );
      em_mat_kost_soll    := coalesce( em_mat_kost_soll,   0 );
      roh_mat_kost_ist    := coalesce( roh_mat_kost_ist,   0 );
      norm_mat_kost_ist   := coalesce( norm_mat_kost_ist,  0 );
      em_mat_kost_ist     := coalesce( em_mat_kost_ist,    0 );
  
      -- Summen
      mat_kost_soll := roh_mat_kost_soll + norm_mat_kost_soll;
      mat_kost_ist  := roh_mat_kost_ist  + norm_mat_kost_ist;
  
      -- Werte von Einzel- und Montageteile müssen immer berücksichtigt werden, Veralteter Stand bis 2024 (siehe #19947):
        -- Einzel- und Montageteile nur bei UE berücksichtigen.
        -- Bei Projekt-NK z.B. nicht. Bei anderer NK aber schon. Fraglich und ggf. mit Option bzgl. Projekt-NK zu lösen.
      --IF _stufe = 1 THEN
      mat_kost_soll := mat_kost_soll + em_mat_kost_soll;
      mat_kost_ist  := mat_kost_ist  + em_mat_kost_ist;
      --END IF;
  

      RETURN;
    END $$ LANGUAGE plpgsql STABLE STRICT;
--

-- Sonderkosten der ABK für Ist nach Stichtag
CREATE OR REPLACE FUNCTION tabk.get_sonder_kosten(
    IN in_abix INTEGER,                  -- ABK
    IN in_dat DATE DEFAULT current_date, -- zum Stichtag
    OUT sond_kost_ist NUMERIC            -- Sonderkosten Ist
    ) RETURNS NUMERIC AS $$
    BEGIN
        -- Sonderkosten entstehen erst nach Abschluss der Fertigung. Spielen lediglich in der Zwischenkalkulation (Ist) und Nachkalkulation eine Rolle.
        sond_kost_ist:=
            COALESCE((SELECT    -- Sonderkosten aus Rechnungspositionen
                        SUM(belp_netto_basis_w)
                      FROM eingrech_pos
                        JOIN eingrech ON beld_id = belp_dokument_id
                      WHERE belp_ab_ix = in_abix -- direkt an ABK
                        AND belp_a2_id IS NULL   -- aber nicht an AG (d.i. Auswärtsbearbeitung)
                        AND beld_erstelldatum <= in_dat) -- nach Stichtag
                , 0)
            +
            COALESCE(tabk.get_sonderkosten_nkz(in_abix) --  EinzelPreis-wirksame Sonderkosten laut NK
                , 0)
            +
            COALESCE(tabk.get_sonderkosten_abg(in_abix) --  EinzelPreis-wirksame Sonderkosten laut AG-KS
                , 0)
        ;
    
        RETURN;
    END $$ LANGUAGE plpgsql STABLE STRICT;
--

-- Sonderkosten laut NK
  -- incl_notInCalc:          inklusive EinzelPreis-unwirksamen Sonderkosten
  -- NOT incl_notInCalc:      nur EinzelPreis-wirksamen Sonderkosten
  -- incl_notInCalc IS NULL:  nur EinzelPreis-unwirksame Sonderkosten
CREATE OR REPLACE FUNCTION tabk.get_sonderkosten_nkz(IN in_abix INTEGER, incl_notInCalc BOOLEAN DEFAULT false) RETURNS NUMERIC AS $$
  DECLARE result NUMERIC(12,4);
  BEGIN
    IF in_abix IS NULL THEN RETURN NULL; END IF;

    result :=
      SUM(nz_stk * nz_ep)
      FROM nkz
      WHERE nz_ab_ix = in_abix
        AND CASE WHEN NOT incl_notInCalc THEN NOT nz_notInCalc  -- nur EinzelPreis-wirksamen Sonderkosten (Standard)
                 WHEN incl_notInCalc     THEN true              -- inklusive EinzelPreis-unwirksamen Sonderkosten
                 ELSE                         nz_notInCalc      -- NULL: nur EinzelPreis-unwirksame Sonderkosten
            END
    ;

    RETURN COALESCE(result, 0);
  END $$ LANGUAGE plpgsql STABLE;
--

-- Sonderkosten laut AG-KS
  -- incl_notInCalc:          inklusive EinzelPreis-unwirksamen Sonderkosten
  -- NOT incl_notInCalc:      nur EinzelPreis-wirksamen Sonderkosten
  -- incl_notInCalc IS NULL:  nur EinzelPreis-unwirksame Sonderkosten
CREATE OR REPLACE FUNCTION tabk.get_sonderkosten_abg(IN in_abix INTEGER, incl_notInCalc BOOLEAN DEFAULT false) RETURNS NUMERIC AS $$
  DECLARE result NUMERIC(12,4);
  BEGIN
    IF in_abix IS NULL THEN RETURN NULL; END IF;

    result :=
      sum(n2_ez_stu *
        CASE WHEN n2_ruest   THEN coalesce( nkk_nssr, ks_stsr ) -- Rüststundensatz
             WHEN n2_pz_para THEN coalesce( nkk_nssm, ks_stsm ) -- Stundensatz für parallele Personalzeit
             ELSE                 coalesce( nkk_nssf, ks_sts )  -- Stundensatz
        END
      )
      FROM nk2
        LEFT JOIN ab2 ON a2_ab_ix = n2_ix AND a2_n = n2_n
        JOIN ksv ON ks_abt = COALESCE(n2_ks, a2_ks)
        LEFT JOIN nkksv ON nkk_ab_ix = n2_ix AND nkk_ks = ks_abt
      WHERE n2_ix = in_abix
        AND NOT n2_ausw
        AND ks_sondkost
        AND CASE WHEN NOT incl_notInCalc THEN NOT ks_notInCalc  -- nur EinzelPreis-wirksamen Sonderkosten (Standard)
                 WHEN incl_notInCalc     THEN true              -- inklusive EinzelPreis-unwirksamen Sonderkosten
                 ELSE                         ks_notInCalc      -- NULL: nur EinzelPreis-unwirksame Sonderkosten
            END
    ;

    RETURN COALESCE(result, 0);
  END $$ LANGUAGE plpgsql STABLE;
--

--
CREATE OR REPLACE FUNCTION tabk.get_aw_kosten(
    _abix             integer,              -- ABK
    _stufe            integer,              -- Soll- und/oder Istdaten, s.o.
    _stichtag         date = current_date,  -- zum Stichtag

    OUT aw_kost_soll  numeric,  -- Auswärtskosten Soll
    OUT aw_kost_ist   numeric   -- Auswärtskosten Ist
    ) RETURNS record AS $$
    DECLARE
        _per_stichtag     boolean;
        aw_best_kost_ist  numeric;
        aw_rech_kost_ist  numeric;
    BEGIN
      -- Auswärtskosten der ABK für Soll und Ist nach Stichtag
        -- Stufe 0: nur Solldaten (Vorkalkulation),
        -- Stufe 1: Soll- und Istdaten ohne NK (UE),
        -- Stufe 2: Soll- und Istdaten inkl. NK (Allgemein),
        -- Stufe 3: nur Istdaten (Nach- und Zwischenkalkulation)
        -- Beachte: ABK-NK-Gemeinkostenzuschläge (nach ABK-Erstellung verfügbar) gelten für alle Stufen.
  
  
      -- Return NULL, wenn ABK nicht existiert.
      IF NOT EXISTS(SELECT true FROM abk WHERE ab_ix = _abix) THEN  RETURN;   END IF;
  
  
      -- Solldaten laut ABK-AG
      -- Gemäß Vorkalkulation aus ASK
      -- Stufen 0,1,2
      IF _stufe <= 2 THEN
          SELECT
            sum(
                -- geplante Fertigungsmenge
                  ab_st_uf1
                -- geplanter AW-Preis
                * a2_awpreis
                -- Auswärtsgemeinkosten
                * ( 1 + ab_nk_agk / 100 )
            )
  
          FROM ab2
            JOIN abk ON a2_ab_ix = ab_ix
          WHERE a2_ab_ix = _abix
            AND a2_ausw
  
          INTO aw_kost_soll
          ;
  
      END IF;
  
  
      -- Istdaten der AW-Bestellungen und AW-Eingangsrechnungen
      -- Gemäß rückgelieferte Menge und Rechnungslegung
      -- Stufen 1,2,3
      IF _stufe >= 1 THEN
          _per_stichtag := coalesce( _stichtag <> current_date, false );  -- mit Konstante für JOIN LATERAL schneller
  
          -- AW-Bestellungen ohne Eingangsrechnung
          SELECT
            sum(
                -- geplanter EP netto inkl. Abzu und Rabatt
                  ld_netto_basis_w / Do1If0( ld_stk_uf1 )
                -- rückgelieferte Menge nach Stichtag
                * coalesce( ld_stkl_fromwendat_todate, ld_stkl )
                -- Auswärtsgemeinkosten
                * ( 1 + ab_nk_agk / 100 )
            )
  
          FROM ldsdok
            JOIN ab2 ON a2_id = ld_a2_id
            JOIN abk ON ab_ix = a2_ab_ix
            -- nach Stichtag
            LEFT JOIN LATERAL teinkauf.ld_stkl_fromwendat_todate( _stichtag, ld_id ) ON _per_stichtag
          WHERE a2_ab_ix = _abix
            AND NOT ld_storno
            -- Bestellung ohne Eingangsrechnung
            AND NOT EXISTS(SELECT true FROM eingrech_pos WHERE belp_a2_id = ld_a2_id)
  
          INTO aw_best_kost_ist
          ;
  
  
          -- AW-Eingangsrechnungen
          SELECT
            sum(
                -- Rechnungsbetrag nach Stichtag
                  tplanterm.ausw_rech_gesamt( a2_id, _stichtag )
                -- Auswärtsgemeinkosten
                * ( 1 + ab_nk_agk / 100 )
            )
  
          FROM ab2
            JOIN abk ON ab_ix = a2_ab_ix
          WHERE a2_ab_ix = _abix
            -- Eingangsrechnung vorhanden
            AND EXISTS(SELECT true FROM eingrech_pos WHERE belp_a2_id = a2_id)
  
          INTO aw_rech_kost_ist
          ;
  
      END IF;
  
      -- wenigstens 0 ausgeben
      aw_kost_soll  := coalesce( aw_kost_soll, 0 );
      aw_kost_ist   := coalesce( aw_best_kost_ist, 0 ) + coalesce( aw_rech_kost_ist, 0 );
  
  
      RETURN;
    END $$ LANGUAGE plpgsql STABLE STRICT;
--


-- Klärung: Verwendung. Ausbauen aus Oberfläche? Nur Addison / Mattis 2012. Nie wirklich in Produktiv gegangen!
-- Aufteilungsbuchungen nach Kostenstellenanteil => EDI Addison Export
CREATE OR REPLACE FUNCTION tabk.abk__kosten__aufteilbuch_pro_ks__by__aknr__nk__last_abix(
    _aknr       VARCHAR,
    OUT ks     VARCHAR,
    OUT anteil NUMERIC,
    OUT mKost  NUMERIC,
    OUT aKost  NUMERIC
    ) RETURNS SETOF RECORD AS $$
    DECLARE
        _r RECORD;
        _lastix  INTEGER;
    BEGIN
  
        --summe Auftragszeit der letzten Nachkalkulation eines Artikels
        _lastix :=
            ld_abk
            FROM ldsdok
            JOIN abk ON ab_ix = ld_abk
            WHERE
                  ld_aknr = _aknr
              AND ab_buch
              AND ab_nk_et IS NOT NULL
            ORDER BY abk.insert_date DESC
            LIMIT 1
        ;
  
        -- nun die aufteilungen
        FOR _r IN SELECT * FROM tabk.abk__kosten__aufteilbuch_pro_ks__by__abix__nk( _lastix ) LOOP
  
            ks     := _r.ks;
            anteil := _r.anteil;
            mkost  := _r.mkost;
            akost  := _r.akost;
  
            RETURN next;
  
        END LOOP;
  
        RETURN;
  
    END $$ LANGUAGE plpgsql STABLE;
  
CREATE OR REPLACE FUNCTION z_99_deprecated.anteilks(
      _aknr      varchar,
      OUT ks     varchar,
      OUT anteil numeric,
      OUT mKost  numeric,
      OUT aKost  numeric
  ) RETURNS SETOF RECORD
  AS $$
    SELECT *
    FROM tabk.abk__kosten__aufteilbuch_pro_ks__by__aknr__nk__last_abix( _aknr )
  $$ LANGUAGE SQL;
-- Aufteilungsbuchungen nach Kostenstellenanteil, nach ABK-Index => EDI Addison Export
CREATE OR REPLACE FUNCTION tabk.abk__kosten__aufteilbuch_pro_ks__by__abix__nk (
    _ix INTEGER,
    OUT ks     VARCHAR,
    OUT anteil NUMERIC,
    OUT mKost  NUMERIC,
    OUT aKost  NUMERIC
    ) RETURNS SETOF RECORD AS $$
    DECLARE
        _sumta   NUMERIC;
        _rec     RECORD;
    BEGIN
  
        --Materialkosten pro Stück, wie in Nachkalk erfasst
        mKost:=
          sum( coalesce( ab_nk_mat, 0 ) )
          FROM abk
          WHERE ab_ix IN ( SELECT * FROM tplanterm.get_all_child_abk( _ix ) )
        ;
  
        mKost := round( mKost, 2 );
        mKost := coalesce( mKost, 0 );
  
        --Auswärtskosten pro Stück, Summe aller Rechnungen auf Auswärtsvergaben
        SELECT
          -- Preis mit Abzuschlägen usw. in Grundmengeneinheit u. Basis_W
          sum( coalesce( belp_abzupreis_gme, belp_preis_gme_basis_w, 0 ) )
        INTO akost
        FROM belegpos
        WHERE
              -- Aller Eingangsrechnungspositionen
              belp_belegtyp = 'ERG'
  
              -- Für diese ABK oder deren Unter-ABKs
          AND belp_a2_id IN (
                SELECT a2_id
                FROM ab2
                WHERE a2_ab_ix IN ( SELECT * FROM tplanterm.get_all_child_abk( _ix ) )
              )
        ;
  
        aKost := coalesce( aKost, 0 );
        aKost := round( aKost, 2 );
  
        _sumta :=
            sum( n2_ez_stu )
            FROM nk2
            WHERE n2_ix IN ( SELECT * FROM tplanterm.get_all_child_abk( _ix ) )
        ;
  
        _sumta := coalesce( _sumta, 0 );
  
        -- Nun die Aufteilungen der gebuchten Stunden laut Nachkalkulation auf die Kostenstellen
        FOR _rec IN (
  
          SELECT
              sum( n2_ez_stu ) AS teil,
              n2_ks
          FROM nk2
          WHERE
                n2_ix IN ( SELECT * FROM tplanterm.get_all_child_abk( _ix ) )
            AND coalesce( n2_ks, '' ) <> ''
          GROUP BY n2_ks
  
        ) LOOP
             ks     := _rec.n2_ks;
             anteil := round( _rec.teil / Do1If0( _sumta ), 4 );
  
             RETURN next;
        END LOOP;
  
        RETURN;
  
    END $$ LANGUAGE plpgsql STABLE;
CREATE OR REPLACE FUNCTION z_99_deprecated.anteilKSix(
    _ix         integer,
    OUT ks     varchar,
    OUT anteil numeric,
    OUT mKost  numeric,
    OUT aKost  numeric
    ) RETURNS SETOF RECORD
    AS $$
        SELECT * FROM tabk.abk__kosten__aufteilbuch_pro_ks__by__abix__nk( _ix )
    $$ LANGUAGE SQL;

 --
 --!kost wieder rausnhemen!
 -- Aufteilungsbuchungen nach Kostenstellenanteil / Vorkalkulation => Für EDI/ADDISON Export
CREATE OR REPLACE FUNCTION tabk.abk__kosten__aufteilbuch_pro_ks__by__abix__opl(
      _ix integer,
      menge integer,
      OUT ks varchar,
      OUT anteil numeric,
      OUT mKost numeric,
      OUT aKost numeric
  ) RETURNS SETOF RECORD
  AS $$
  DECLARE

      _sumpreis  numeric;
      _rkost     numeric;
      _fm        integer;
      _lastix    integer;
      _rec       record;
  BEGIN

      _lastix := _ix;
      _fm     := menge;

      --Materialkosten pro Stück * Menge
      mkost := round(
                 coalesce( TArtikel.op6_kost( _lastix, _fm ) , 0 ) * _fm
                 , 4
               );

      --Auswärtskosten pro Stück * Menge
      akost := round(
                 coalesce( TArtikel.op2_akost( _lastix, _fm ) , 0 ) * _fm
                 , 4
               );

      --Rüstkosten pro Stück
      _rKost :=
            round(
                coalesce(
                    sum(
                        coalesce(
                            o2_tr_sek / 3600 * ks_stsr ,
                            0
                        )
                        * coalesce(
                              ( 1 + op_rgk / 100 ),
                              1
                          )
                    ),
                    0
                ),
                4
            )
            FROM op2, ksv, opl
            WHERE
                o2_ix = $1
            AND op_ix = o2_ix
            AND ks_abt = o2_ks
            AND NOT o2_aw
      ;

      -- Gesamtpreis = ((Maschinenkosten/Stück + Rüstkosten/Stück) * Menge)
      _sumpreis := ( TArtikel.op2_mkost( _lastix, _fm ) + _rkost ) * _fm;

      --Aufteilungen nach Zeit*Stundensatz pro Kostenstelle
      FOR _rec IN

          SELECT
              sum(
                ifthen(
                  o2_aw,
                  0,
                   --Kosten eigene KS, analog o2_mkost*menge + o2__rkost*menge, aber nur 1 KS
                  (
                    coalesce(
                        ( o2_th_sek + o2_tn_sek ) * ( 1 + o2_tv / 100 ) / 3600 * ks_sts,
                        0
                    )
                    * coalesce(
                          ( 1 + op_fgk / 100 ),
                          1
                    )
                  )
                  * _fm
                  + (
                    coalesce(
                        o2_tr_sek / 3600 * ks_stsr,
                        0
                    )
                    * coalesce(
                        ( 1 + op_rgk / 100 ),
                        1
                      )
                    )
                    * _fm
                )
              ) AS teil,
              o2_ks
          FROM op2, opl, ksv
          WHERE
                o2_ix = _lastix
            AND op_ix = o2_ix
            AND ks_abt = o2_ks
            AND coalesce( o2_ks, '' ) <> ''
          GROUP BY o2_ks
      LOOP

          ks := _rec.o2_ks;
          anteil := round( _rec.teil / _sumpreis, 4 );
          RETURN next;

      END LOOP;

      RETURN;

  END $$ LANGUAGE plpgsql STABLE;
CREATE OR REPLACE FUNCTION z_99_deprecated.anteilKSixVorkalk(
    _ix         integer,
    _menge      integer,
    OUT ks     varchar,
    OUT anteil numeric,
    OUT mKost  numeric,
    OUT aKost  numeric
    ) RETURNS SETOF RECORD
    AS $$
        SELECT * FROM tabk.abk__kosten__aufteilbuch_pro_ks__by__abix__opl( _ix, _menge )
    $$ LANGUAGE SQL;
--

-- #20419 Diese Funktion erzeugt Messdatensätze anhand einer Liste laufender Nummern, die der Anwender als String übergibt.
-- Der Eingabestring kann kommasepariert einzelne Nummern oder auch ganze Bereiche enthalten.
-- Beispiel: '20, 21, 30-35, 40 ,   41'
CREATE OR REPLACE FUNCTION tabk.oplpm_mw__insert_from__inputstring( _pm_id integer, _input varchar ) RETURNS void AS $$
DECLARE
  _lfd_nrn integer[];
BEGIN

  -- Ohne gültige pm_id gibt es nichts zu tun.
  IF NOT EXISTS( SELECT 1 FROM oplpm_data WHERE pm_id = _pm_id ) THEN
    RETURN;
  END IF;

  -- Eingabestring zu Liste von einzelnen Teilenummern erweitern
  _lfd_nrn := integer_list__from__inputstring__get( _input );

  -- Alle laufenden Nummern der Liste werden dem Messwertprotokoll hinzugefügt.
  -- Duplikate werden dabei übergangen.
  INSERT INTO oplpm_mw ( mw_pm_id, mw_lfdnr )
  SELECT _pm_id, unnest FROM unnest( _lfd_nrn )
  LEFT JOIN oplpm_mw ON mw_pm_id = _pm_id AND mw_lfdnr = unnest
  WHERE mw_id IS null;

END $$ LANGUAGE plpgsql STRICT;
--


CREATE OR REPLACE FUNCTION tabk.messteile__prozentual__get(
  _anzahl_teile integer,
  _pruefanteil_prozent integer
) RETURNS SETOF integer AS $$
  DECLARE
    _schrittweite numeric;
    _anzahl integer;
  BEGIN

    -- Funktion zur Ermittlung der zu testenden Teile bei prozentualer Prüfanteil
    -- Es wird stets mit dem Teil 1 begonnen
    -- Beispiel: Fertigungsmenge 10 und 50% Prüfanteil ergibt die zu prüfenden Teile 1, 3, 5, 7 und 9

    IF _pruefanteil_prozent IS null OR _pruefanteil_prozent <= 0 THEN
      RETURN;
    END IF;
    IF _pruefanteil_prozent > 100 THEN
      _pruefanteil_prozent := 100;
    END IF;

    _schrittweite := 100.0 / _pruefanteil_prozent;

    -- Anzahl der zu prüfenden Teile stets aufrunden, um Prüfquote zu erfüllen
    _anzahl := ceil( _anzahl_teile * _pruefanteil_prozent / 100.0 )::integer;
    RETURN QUERY
      SELECT DISTINCT min( ceil(x * _schrittweite + 1)::integer, _anzahl_teile )::integer
      FROM generate_series(0, _anzahl-1) AS x;

  END $$ LANGUAGE plpgsql IMMUTABLE;


CREATE OR REPLACE FUNCTION tabk.messteile__abk__get(
  _anzahl_teile integer,
  _pruefhaeufigkeit integer,
  _pruefanteil_prozent integer
) RETURNS SETOF integer AS $$
  BEGIN

    -- ermittelt die zu prüfenden Teile in Abhängigkeit von
    --   der Fertigungsmenge
    --   der kodierten Prüfhäufigkeit
    --   ggf. dem prozentualen Prüfanteil

    IF _anzahl_teile IS null OR _anzahl_teile <= 0 THEN
      RETURN;
    END IF;

    CASE _pruefhaeufigkeit
      WHEN 10 THEN -- 'Prozentual';
        RETURN QUERY SELECT tabk.messteile__prozentual__get( _anzahl_teile, _pruefanteil_prozent );
      WHEN 20 THEN -- 'Erst- Letztteil'
        RETURN NEXT 1;
        IF _anzahl_teile > 1 THEN
          RETURN NEXT _anzahl_teile;
        END IF;
      WHEN 30 THEN  -- 'Erstteil'
        RETURN NEXT 1;
      WHEN 40 THEN  -- 'Letzteil'
        RETURN NEXT _anzahl_teile;
      WHEN 50 THEN  -- 'Prozentual und Letztteil'
        RETURN QUERY
          SELECT * FROM (
            SELECT tabk.messteile__prozentual__get( _anzahl_teile, _pruefanteil_prozent )
            UNION
            SELECT 1
            UNION
            SELECT _anzahl_teile
          ) AS zahlen
          ORDER BY 1;
      ELSE
    END CASE;

  END $$ LANGUAGE plpgsql IMMUTABLE;

CREATE OR REPLACE FUNCTION tabk.oplpm_mw__insert_from__oplpm_data(
  _pm_id integer,
  _letztteil integer = null
) RETURNS void AS $$
  DECLARE
    _anzahl_teile integer;
    _pruefhaeufigkeit integer;
    _pruefanteil_prozent integer;
  BEGIN

  -- Messdatenerfassung, Prüfplan
  -- Erzeugt anhand der Einstellungen wie Prüfhäufigkeit (pm_intv_typ) und Prüfintervall (pm_pi) sowie der zu produziernden Menge eine Liste leerer Messwerte,
  -- welche es später zu füllen gilt.

  _anzahl_teile := coalesce( _letztteil, ( SELECT FertMenge FROM TABK.oplpm__pi__get_anzahl( _pm_id )));
  SELECT pm_intv_typ, asNumeric( pm_pi ) INTO _pruefhaeufigkeit, _pruefanteil_prozent FROM oplpm_data WHERE pm_id = _pm_id;

  IF
    -- Nur für prozentuale Häufigkeit und Erst-/Letztteile werden Messwertdatensätze automatisch erzeugt,
    -- und das auch nur dann, wenn ein
    _pruefhaeufigkeit = ANY( ARRAY[ 10,20,30,40,50 ])
  THEN
    -- #20681 Erzeugung von Messdatensätzen nur, wenn keine zur laufenden Nummer vorhanden
    INSERT INTO oplpm_mw
      ( mw_pm_id, mw_lfdnr                                                                             )
    SELECT _pm_id, messteile__abk__get FROM tabk.messteile__abk__get( _anzahl_teile, _pruefhaeufigkeit, _pruefanteil_prozent )
    EXCEPT
    SELECT _pm_id, mw_lfdnr FROM oplpm_mw WHERE mw_pm_id = _pm_id;
  END IF;

  END $$ LANGUAGE plpgsql;
  --


    CREATE OR REPLACE FUNCTION tabk.oplpm_data_pruefmenge__anzahl__get(
        _opmpmg_id         integer,    -- Verweis auf Prüfhäufigkeit
        _anzahl_teile      integer,    -- Anzahl der gefertigten Teile
        _anteil_in_prozent numeric     -- Anteil der zu prüfenden Teile in Prozent
    ) RETURNS integer AS $$
      DECLARE
          _erstteil     boolean;
          _letztteil    boolean;
          _prozent      boolean;
          _anzahl_pruef integer;
      BEGIN

      -- ermittelt die Anzahl der zu prüfenden Teile

      _erstteil := false;
      _letztteil:= false;
      _prozent  := false;

      SELECT opmpmg_erstteil, opmpmg_letztteil, opmpmg_prozent
        INTO       _erstteil,       _letztteil,       _prozent
      FROM oplpm_data_pruefmenge WHERE opmpmg_id = _opmpmg_id;

      _anzahl_pruef := 0;

      IF _erstteil THEN
        _anzahl_pruef := _anzahl_pruef + 1;
      END IF;

      IF _letztteil THEN
        _anzahl_pruef := _anzahl_pruef + 1;
      END IF;

      IF _prozent THEN
        _anzahl_pruef := _anzahl_pruef + max( ceil( _anzahl_teile * _anteil_in_prozent / 100.0 ), 0 );
      END IF;

      RETURN _anzahl_pruef;

    END $$ LANGUAGE plpgsql STABLE;
    ----

    CREATE OR REPLACE FUNCTION TABK.oplpm__pi__get_anzahl(IN in_pm_id integer, OUT anzpruefsoll integer, OUT anzpruefist integer, OUT fertmenge integer)
     RETURNS record
     LANGUAGE plpgsql
     STABLE
    AS $function$
      DECLARE rec RECORD;
        mypm_a2_id    INTEGER;
        mypm_intv_typ   INTEGER;
        mypm_pi     INTEGER;
     BEGIN
       AnzPruefSoll = 0;
       AnzPruefIst = 0;
       FertMenge = 0;
       IF (SELECT pm_a2_id FROM oplpm_data WHERE pm_id = in_pm_id) > 0 THEN
            mypm_a2_id = (SELECT COALESCE(pm_a2_id, 0) FROM oplpm_data WHERE pm_id = in_pm_id);

            -- #20386 Es soll immer das Maximum aus Fertigungs- und Sollmenge für die Messpunkterzeugung maßgebend sein.
            FertMenge = ( SELECT max( ab_st_uf1_soll , coalesce( ab_st_uf1, 1 )) FROM ab2 JOIN abk ON ab_ix = a2_ab_ix WHERE a2_id = mypm_a2_id )::integer;
       ELSE
            FertMenge = (SELECT COALESCE(op_lg, 0) FROM oplpm_data JOIN op2 ON o2_id = pm_op2_id JOIN opl ON op_ix = o2_ix WHERE pm_id = in_pm_id);
       END IF;

       IF FertMenge IS NULL OR FertMenge = 0 THEN
          FertMenge = 1;
       END IF;
       AnzPruefIst = (SELECT COUNT(*) FROM oplpm_mw WHERE mw_pm_id = in_pm_id);

       mypm_intv_typ = (SELECT pm_intv_typ FROM oplpm_data WHERE pm_id = in_pm_id);
        --1-Prozentual, 2-Erst- Letztteil, 3-Erstteil, 4-Letzteil, ..., 100-Andere
       mypm_pi = (SELECT AsNumeric(pm_pi, true) FROM oplpm_data WHERE pm_id = in_pm_id);

       AnzPruefSoll := tabk.oplpm_data_pruefmenge__anzahl__get( mypm_intv_typ, FertMenge, mypm_pi::numeric );

       RETURN;
    END $function$;
    -----
    CREATE OR REPLACE FUNCTION tabk.TeilPruef_IntervalBez(IN typNummer INTEGER) RETURNS VARCHAR AS $$
     BEGIN
       CASE typNummer
          WHEN 1 THEN
             RETURN Lang_Text(21285);   --  'Prozentual';
          WHEN 2 THEN
             RETURN Lang_Text(29059);   --  'Erst- Letztteil';
          WHEN 3 THEN
             RETURN Lang_Text(29060);   --  'Erstteil';
          WHEN 4 THEN
             RETURN Lang_Text(29061);   --  'Letzteil';
          WHEN 100 THEN
             RETURN Lang_Text(29062);   --  'Freie Eingabe';
          ELSE
             RETURN Lang_Text(12332);   --  'Unbekannt';
       END CASE;
    END $$ LANGUAGE plpgsql VOLATILE;
-----


CREATE OR REPLACE FUNCTION tabk.abk__stv__is(_ab_ix integer) RETURNS boolean
    AS $$
       SELECT EXISTS(SELECT true FROM abk WHERE ab_parentabk = _ab_ix)
    $$ LANGUAGE sql STABLE PARALLEL SAFE;

-- Materialnachweis (effektiv verbrauchtes Material der ABK) #6549, #7513
-- Beachte tplanterm.get_all_child_abk mit Option OnlyByLifsch
-- Bsp. unten
CREATE OR REPLACE FUNCTION tabk.abk_lifsch_materialnachweis(
    INOUT   abix            INTEGER,    -- ABK
    INOUT   agid            INTEGER DEFAULT NULL,   -- ID Materialposition (wenn vorhanden), Einschränkung auf angg. Materialposition möglich
    IN      mit_matpos      BOOLEAN DEFAULT true,   -- Ausgabe ist mit Buchungen an Materialpositionen, false: nur freie Buchungen ohne Bezug zu Materpositionen
    IN      mit_chnr_sernr  BOOLEAN DEFAULT true,   -- Ausgabe ist mit Ermittlung von Chargen- und Seriennummern (abstellbar zwecks Performance, wenn nicht benötigt).
    OUT     mat_pos         INTEGER,    -- Materialposition (wenn vorhanden)
    OUT     mat_aknr        VARCHAR,    -- Material
    OUT     lnr             INTEGER,    -- pro Lagerabgang, kann dann je nach Anwendung pro Materialposition oder Artikel aggregiert werden.
    OUT     is_effektiv     BOOLEAN,    -- Menge usw. ist effektiv vorhanden (bei kompletter Rückführung false). Beachte true bei Mindermengenbuchung mit 0 (Charge relevant).
    OUT     abgg_effektiv   NUMERIC,    -- effektiv gebuchte Menge (Abgang-Rückführung)
    OUT     chnr_effektiv   VARCHAR,    -- effektiv verwendete Chargennummer (bei kompletter Rückführung leer)
    OUT     sernr_effektiv  VARCHAR[],  -- effektiv verwendete Seriennummern, Array kann dann je nach Anwendung mit array_to_string ausgegeben werden.
    OUT     idx_effektiv    VARCHAR,    -- Zeichnungsindex der effektiv verwendeten Charge
    OUT     menge_abgang    NUMERIC,    -- l_abgg_uf1, keine Summe, siehe lnr
    OUT     chnr_abgang     VARCHAR,    -- verwendete Charge bei Abgang
    OUT     sernr_abgang    VARCHAR[],  -- Seriennummern des Abgangs
    OUT     menge_rckbuch   NUMERIC,    -- Summe w_zugang_uf1
                                        -- neu verwendete Chargen der Rückführungen sind irrelvant, da Rückführung auf Abgang gebucht wird (Charge des Abgangs führend).
    OUT     sernr_rckbuch   VARCHAR[],  -- Seriennummern der Rückführungen
    OUT     ldid            INTEGER,    -- Bestellbezug der ausgebuchten Lagermenge
    OUT     own_abk         INTEGER     -- ABK des Unterbauteils (Unter-ABK)
  ) RETURNS SETOF RECORD AS $$
  DECLARE lifsch_rec RECORD;
          wendat_rec RECORD;
  BEGIN
    IF abix IS NULL AND agid IS NULL THEN RETURN; END IF; -- beide Eingabeparamter NULL, dann raus

    FOR lifsch_rec IN -- Alle Lagerabgänge der ABK oder der zug. Materialposition
        SELECT l_ab_ix, l_nr, l_abgg_uf1, l_aknr, l_lgchnr, l_w_wen,
               ld_id, coalesce(nullif(ld_abk, -1), ag_ownabk) AS own_abk, -- besondere Verwendung von -1 siehe ldsdok
               ag_pos, ag_id
          FROM lifsch
          LEFT JOIN ldsdok ON ld_id = l_ld_id -- Lagermenge kommt aus Bestellung
          LEFT JOIN auftg ON mit_matpos AND ag_id = l_ag_id -- auch freie LA
         WHERE (abix IS NULL OR l_ab_ix = abix) -- mgl. Einschränkung auf ABK
           AND (agid IS NULL OR ag_id = agid) -- und/oder mgl. Einschränkung auf bestimmte Matpos
           AND (mit_matpos OR l_ag_id IS NULL) -- nur freie Buchungen ohne Bezug zu Matpos
               -- Lagerbuchungen UE sind nicht relevant für die ABK Materialnachweis
           AND NOT TSystem.ENUM_GetValue(ag_stat, 'UE')
         ORDER BY
               ag_pos NULLS LAST, l_aknr, l_nr -- erst positionsbezogene, dann freie, sortiert nach Artikelnummer, LNr.
    LOOP
        abix:=          lifsch_rec.l_ab_ix;
        agid:=          lifsch_rec.ag_id;
        mat_pos:=       lifsch_rec.ag_pos;
        mat_aknr:=      lifsch_rec.l_aknr;
        lnr:=           lifsch_rec.l_nr;
        --menge_abgang:=  lifsch_rec.l_abgg_uf1;
        abgg_effektiv:=  lifsch_rec.l_abgg_uf1;
        ldid:=          lifsch_rec.ld_id;
        own_abk:=       lifsch_rec.own_abk;

        IF mit_chnr_sernr THEN
            chnr_abgang:=  lifsch_rec.l_lgchnr;
            sernr_abgang:= array_agg(lgs_sernr ORDER BY lgs_sernr) FROM lagsernr WHERE lgs_l_nr = lnr; -- alle Seriennummern des Abgangs
        END IF;

        wendat_rec:= NULL;

        -- zum Lagerabgang zugehörige Materialrückführungen
        SELECT SUM(w_zugang_uf1) AS sum_rckbuch, array_agg(w_wen) AS w_wens
        INTO wendat_rec
        FROM wendat
        WHERE w_l_nr = lnr;

        IF mit_chnr_sernr THEN
            sernr_rckbuch:= array_agg(lgs_sernr) FROM lagsernr WHERE lgs_w_wen = ANY (wendat_rec.w_wens); -- Alle Seriennummern aller Rückführungen. Separat wegen array_agg auf SubSelect.
        END IF;

        menge_rckbuch:= wendat_rec.sum_rckbuch;
        --abgg_effektiv:= menge_abgang - COALESCE(menge_rckbuch, 0);
        menge_abgang:= abgg_effektiv + COALESCE(menge_rckbuch, 0);

        IF abgg_effektiv > 0 OR (abgg_effektiv = 0 AND menge_rckbuch IS NULL) THEN -- Es gibt effektive Menge oder Mindermengenbuchung ohne Rückführung
            is_effektiv:= true;
            IF mit_chnr_sernr THEN
                idx_effektiv:= ld_aknr_idx FROM ldsdok JOIN wendat ON w_lds_id = ld_id WHERE w_wen = lifsch_rec.l_w_wen; -- Zeichnungsindex der abgebuchten Lagercharge
                chnr_effektiv:= chnr_abgang;
                IF sernr_rckbuch IS NOT NULL THEN -- gibt es rückgeführte Seriennummern,
                    -- dann alle Seriennummern welche nicht rückgeführt sind, quasi als Differenz, neu in Array.
                    sernr_effektiv:= array_agg(l_sern ORDER BY l_sern) FROM (SELECT unnest(sernr_abgang) AS l_sern) AS sub WHERE l_sern <> ALL (sernr_rckbuch);
                ELSE
                    sernr_effektiv:= sernr_abgang;
                END IF;
            END IF;
        ELSE
            is_effektiv:= false;
            idx_effektiv:= NULL;
            chnr_effektiv:= NULL; -- oder lieber ''
            sernr_effektiv:= NULL; -- oder lieber ARRAY[''::VARCHAR]
        END IF;

        RETURN NEXT;
    END LOOP;
  END $$ LANGUAGE plpgsql STABLE;
--

/* Verwendung Materialnachweis mit Bsp für Aggregat-Bildung inkl. to_string
    SELECT
      abix, mat_pos, mat_aknr,
      l_abgg_effektiv,
      is_effektiv, -- Position ist effektiv.
      -- effektiv verwendete Chargen
      array_to_string(chnr_effektiv, ', ') AS l_lgchnr, -- Sortierung im Array bleibt erhalten.
      -- effektiv verwendete Seriennummern
      (SELECT string_agg(DISTINCT sern, ', ' ORDER BY sern) FROM (SELECT unnest(sernr_effektiv) AS sern) AS sub) AS lgs_sernr -- Sortierung über aggregierte Arrays muss neu gemacht werden.
    FROM (
        SELECT
          abix, mat_pos, mat_aknr,
          bool_or(is_effektiv) AS is_effektiv, -- Ist einer der aggregierten LA effektiv? Dann Pos. effektiv.
          SUM(abgg_effektiv) AS l_abgg_effektiv,
          array_agg(DISTINCT nullif(chnr_effektiv, '') ORDER BY nullif(chnr_effektiv, '')) AS chnr_effektiv, -- Als Array mit Sortierung aggregieren.
          array_cat_agg(sernr_effektiv) AS sernr_effektiv -- Arrays der Seriennummern in Array aggregieren. Sortierung hier nicht ohne Weiteres möglich.
        FROM (
            SELECT DISTINCT get_all_child_abk AS childs FROM tplanterm.get_all_child_abk(:ab_ix, false, false, true) -- Nur anhand Buchungen (OnlyByLifsch)
        ) AS child_abks
          JOIN LATERAL tabk.abk_lifsch_materialnachweis(childs) ON true
        GROUP BY abix, mat_pos, mat_aknr
    ) AS aggregate_pro_matpos_aknr
    -- ggf. JOINS
    ORDER BY abix, mat_pos NULLS LAST, mat_aknr -- freie Zubuchungen auf ABK ans Ende, sortiert nach Artikel
    ;
*/

-- ABK-Splittung #7812
CREATE OR REPLACE FUNCTION tabk.abk_split(IN inSrc_ABK INTEGER, IN split_menge NUMERIC) RETURNS VOID AS $$
  DECLARE abk_src     RECORD;
          split_abk   INTEGER;
  BEGIN
    IF inSrc_ABK IS NULL OR split_menge IS NULL THEN RETURN; END IF; -- keine korrekte Eingabe, dann raus

    SELECT abk.*, ld_aknr, ld_akbz, COALESCE(ld_stkl, 0) AS ld_stkl INTO abk_src FROM abk LEFT JOIN ldsdok ON ld_id = ab_ld_id WHERE ab_ix = inSrc_ABK; -- ABK-Daten holen

    IF abk_src IS NULL THEN RETURN; END IF; -- keine Quelldaten, dann raus

    IF (COALESCE(abk_src.ab_st_uf1_soll, abk_src.ab_st_uf1) - abk_src.ld_stkl - split_menge) <= 0 THEN -- Fehler bei unsinniger Mengenangabe.
        RAISE EXCEPTION '%', lang_text(16644); -- Keine Splittung möglich...
    END IF;

    -- Hinweise
        IF NOT EXISTS(SELECT true FROM ab2 WHERE NOT a2_ende AND a2_ab_ix = inSrc_ABK) THEN -- Es gibt keine offenen AG
            PERFORM PRODAT_TEXT(16645);
        END IF;

        IF EXISTS(SELECT true FROM auftg WHERE ag_parentabk = inSrc_ABK AND ag_ownabk IS NOT NULL AND NOT ag_done) THEN -- Es gibt offene ABK von Unterbauteilen, die ggf. auch gesplittet werden müssen.
            PERFORM PRODAT_TEXT(16646);
        END IF;
    --

    -- ABK 1 mit restlicher Produktionsmenge
        INSERT INTO abk (
            ab_ap_nr, ab_ap_bem,
            ab_parentabk,
            ab_stat,
            ab_st_uf1, ab_st_uf1_soll,
            ab_ap_txt, ab_ap_txt_rtf, ab_mainabk, ab_tablename, ab_keyvalue, ab_dbrid, ab_an_nr,
            ab_askix)
        SELECT
            COALESCE(abk_src.ab_ap_nr, abk_src.ld_aknr), COALESCE(abk_src.ab_ap_bem, abk_src.ld_akbz), -- Artikel und ggf. überschriebene Artikelbez. übertragen
            inSrc_ABK, -- unter Quell-ABK hängen
            'SPL', -- Kennzeichnung Splitt-ABK
            abk_src.ab_st_uf1 - abk_src.ld_stkl - split_menge, abk_src.ab_st_uf1_soll - abk_src.ld_stkl - split_menge, -- bereits gelieferte Menge und Splitt-Menge abziehen
            abk_src.ab_ap_txt, abk_src.ab_ap_txt_rtf, abk_src.ab_mainabk, abk_src.ab_tablename, abk_src.ab_keyvalue, abk_src.ab_dbrid, abk_src.ab_an_nr, -- Quelldaten 1:1 in Zieldaten
            -1 -- sonst wird automatisch Standard-ASK des Artikels geholt und die ABK mit entspr AG und Material gefüllt
        RETURNING ab_ix INTO split_abk;

        -- AG kopieren
        PERFORM copyAG_toABK(split_abk, inSrc_ABK);

        -- Ziel-AG entspr. AG-Rangfolge in Quell-ABK auf Erledigt setzen.
        -- AG-Nr. entspr. Quell-ABK setzen.
        -- Beim Kopieren werden AG-Nr. nicht mitkopiert, sondern beginnen wieder ab 10. Daher keine Identifikation über AG-Nr. möglich, sondern nur per Rangfolge.
        UPDATE ab2 SET a2_n = -a2_n WHERE a2_ab_ix = split_abk; -- Vorher mögliche Konflikte vermeiden, wegen UNIQUE. Dann ORDER BY unten aber DESC.

        UPDATE ab2 SET
          a2_ende = ab2_src.a2_ende,
          a2_n    = ab2_src.a2_n
        FROM (
            SELECT
              (   SELECT ab2__split_abk.a2_id
                  FROM ab2 AS ab2__split_abk
                  WHERE ab2__split_abk.a2_ab_ix = split_abk
                  ORDER BY ab2__split_abk.a2_n DESC OFFSET rown - 1 LIMIT 1 -- ID des Ziel-AG anhand gleichem Rang wie AG aus Quell-ABK
              ) AS split_abk__a2_id,
              a2_n, a2_ende
            FROM (
                SELECT row_number() OVER(ORDER BY a2_n) AS rown, a2_n, a2_ende -- Rang, AG-Nr. und Status der AG aus Quell-ABK
                FROM ab2 WHERE a2_ab_ix = inSrc_ABK
            ) AS sub
        ) AS ab2_src
        WHERE a2_ab_ix = split_abk
          AND a2_id = split_abk__a2_id
        ;

        -- ASK-Index nachsetzen
        IF EXISTS(SELECT true FROM ab2 WHERE a2_ab_ix = split_abk) THEN -- aber nur wenn es AG gibt, sonst werden automatisch AG/Mat aus der ASK geholt.
            UPDATE abk SET ab_askix = abk_src.ab_askix WHERE ab_ix = split_abk;
        END IF;
        --
    --

    -- ABK 2 mit Splitt-Menge
        split_abk:= NULL;

        INSERT INTO abk (
            ab_ap_nr, ab_ap_bem,
            ab_parentabk,
            ab_stat,
            ab_st_uf1, ab_st_uf1_soll,
            ab_ap_txt, ab_ap_txt_rtf, ab_mainabk, ab_tablename, ab_keyvalue, ab_dbrid, ab_an_nr,
            ab_askix)
        SELECT
            COALESCE(abk_src.ab_ap_nr, abk_src.ld_aknr), COALESCE(abk_src.ab_ap_bem, abk_src.ld_akbz), -- Artikel und ggf. überschriebene Artikelbez. übertragen
            inSrc_ABK, -- unter Quell-ABK hängen
            'SPL', -- Kennzeichnung Splitt-ABK
            split_menge, NULL, -- Splitt-Menge eintragen. Es gibt keine Sollmenge/geplanten Ausschuss bei der ABK 2, bleibt alles bei ABK 1.
            abk_src.ab_ap_txt, abk_src.ab_ap_txt_rtf, abk_src.ab_mainabk, abk_src.ab_tablename, abk_src.ab_keyvalue, abk_src.ab_dbrid, abk_src.ab_an_nr,
            -1 -- sonst wird automatisch Standard-ASK des Artikels geholt und die ABK mit entspr AG und Material gefüllt
        RETURNING ab_ix INTO split_abk;

        -- AG kopieren
        PERFORM copyAG_toABK(split_abk, inSrc_ABK);

        -- Ziel-AG entspr. AG-Rangfolge in Quell-ABK auf Erledigt setzen.
        -- AG-Nr. entspr. Quell-ABK setzen.
        -- Beim Kopieren werden AG-Nr. nicht mitkopiert, sondern beginnen wieder ab 10. Daher keine Identifikation über AG-Nr. möglich, sondern nur per Rangfolge.
        UPDATE ab2 SET a2_n = -a2_n WHERE a2_ab_ix = split_abk; -- Vorher mögliche Konflikte vermeiden, wegen UNIQUE. Dann ORDER BY unten aber DESC.

        UPDATE ab2 SET
          a2_ende = ab2_src.a2_ende,
          a2_n    = ab2_src.a2_n
        FROM (
            SELECT
              (   SELECT ab2__split_abk.a2_id
                  FROM ab2 AS ab2__split_abk
                  WHERE ab2__split_abk.a2_ab_ix = split_abk
                  ORDER BY ab2__split_abk.a2_n DESC OFFSET rown - 1 LIMIT 1 -- ID des Ziel-AG anhand gleichem Rang wie AG aus Quell-ABK
              ) AS split_abk__a2_id,
              a2_n, a2_ende
            FROM (
                SELECT row_number() OVER(ORDER BY a2_n) AS rown, a2_n, a2_ende -- Rang, AG-Nr. und Status der AG aus Quell-ABK
                FROM ab2 WHERE a2_ab_ix = inSrc_ABK
            ) AS sub
        ) AS ab2_src
        WHERE a2_ab_ix = split_abk
          AND a2_id = split_abk__a2_id
        ;

        -- ASK-Index nachsetzen
        IF EXISTS(SELECT true FROM ab2 WHERE a2_ab_ix = split_abk) THEN -- aber nur wenn es AG gibt, sonst werden automatisch AG/Mat aus der ASK geholt.
            UPDATE abk SET ab_askix = abk_src.ab_askix WHERE ab_ix = split_abk;
        END IF;
        --
    --

    -- Quell-ABK
        -- AG beenden
        UPDATE ab2 SET a2_ende = true WHERE a2_ab_ix = inSrc_ABK AND NOT a2_ende;
    --

    RETURN;
  END $$ LANGUAGE plpgsql;
--id , parent , identname , ident , subject , insertby , usename , dat, txt
CREATE OR REPLACE FUNCTION ShowABKAndStructure(IN _tablename VARCHAR, IN _dbrid VARCHAR, IN a2id VARCHAR, OUT id VARCHAR, OUT parent VARCHAR, OUT ismyrec BOOL, OUT identname VARCHAR(10), OUT ident VARCHAR(50), OUT subject VARCHAR(1000), OUT insertby VARCHAR(50), OUT usename VARCHAR(50), OUT dat DATE, OUT txt TEXT, OUT done BOOL)  RETURNS SETOF RECORD AS $$
    DECLARE projnr VARCHAR;
            projdbrid VARCHAR;
            kanfnr VARCHAR;
            s VARCHAR;
            _abix VARCHAR;
            B BOOLEAN;
            r RECORD;
            r1 RECORD;
            r2 RECORD;
            r3 RECORD;
            r4 RECORD;
    --Idents:
    -- -1 -> Projekt
    BEGIN
     ismyrec:=FALSE;
     --
     IF _tablename='anl' THEN
            SELECT dbrid, an_nr INTO projdbrid, projnr FROM anl WHERE dbrid=_dbrid;
     END IF;
     --
     CREATE TEMP TABLE tmpdata (pos INTEGER, tname VARCHAR(100), kanf VARCHAR(100), tdbrid VARCHAR(100));
     --Servicevorfall (QAB) <-> ServiceANF
     IF _tablename='qab' THEN
            SELECT kanf_an_nr, kanf_nr INTO projnr, kanfnr FROM qab JOIN kundservicepos ON q_kanfse_id=kanfse_id JOIN kundanfrage ON kanf_nr=kanfse_kanf_nr WHERE _dbrid=qab.dbrid;
     END IF;
     --ServiceANF
     IF _tablename='kundservicepos' THEN
            SELECT kanf_an_nr, kanf_nr INTO projnr, kanfnr FROM kundservicepos JOIN kundanfrage ON kanf_nr=kanfse_kanf_nr WHERE kundservicepos.dbrid=_dbrid;
     END IF;
     --
     IF projnr IS NOT NULL THEN
            SELECT an_bez, dbrid INTO subject, projdbrid FROM anl WHERE an_nr=projnr;

            id:=projdbrid;
            identname:='anl.an_nr';
            ident:=projnr;
            RETURN NEXT;
            INSERT INTO tmpdata(pos, tname, tdbrid) VALUES (99999, 'anl', projdbrid);
            --alle Anfragen anzeigen
            FOR r IN SELECT kanf_nr, kanf_date, kanf_isservice, dbrid, kanf_absdate FROM kundanfrage WHERE kanf_an_nr=projnr AND kanf_nr<>kanfnr ORDER BY kanf_date DESC LOOP
                    id:=r.kanf_nr;
                    parent:=projdbrid;
                    identname:=IFTHEN(r.kanf_isservice, 'anfrageservice', 'kundanfrage');
                    ident:=r.kanf_nr;
                    subject:=r.kanf_nr;
                    dat:=r.kanf_date;
                    done:=r.kanf_absdate IS NOT NULL;
                    RETURN NEXT;
                    INSERT INTO tmpdata(kanf) VALUES (r.kanf_nr);
            END LOOP;
     END IF;
     --eigentliche Anfrage
     SELECT kanf_date, kanf_isservice, kanf_absdate INTO r  FROM kundanfrage WHERE kanf_nr=kanfnr;
     id:=kanfnr;
     parent:=projdbrid;
     identname:=IFTHEN(r.kanf_isservice, 'anfrageservice.kanf_nr', 'kundanfrage.kanf_nr');
     subject:=kanfnr;
     ident:=kanfnr;
     dat:=NULL;
     done:=r.kanf_absdate IS NOT NULL;
     RETURN NEXT;
     INSERT INTO tmpdata(kanf) VALUES (kanfnr);
     --Fehlereingänge
     FOR r IN SELECT kanf, kundservicepos.dbrid AS spdbrid, kanfse_pos, kanfse_sfall, q_nr, qab.dbrid AS qabdbrid FROM kundservicepos LEFT JOIN qab ON q_kanfse_id=kanfse_id, tmpdata WHERE kanfse_kanf_nr=kanf ORDER BY kanfse_pos LOOP
            id:=r.spdbrid;
            parent:=r.kanf;
            identname:='kundservicepos';
            ident:=r.kanfse_pos;
            subject:=r.kanfse_sfall;
            dat:=NULL;
            done:=NULL;
            RETURN NEXT;
            INSERT INTO tmpdata(tname, tdbrid) VALUES ('kundservicepos', r.spdbrid);
            --qab?
            --RAISE EXCEPTION 'x';
            IF r.spdbrid IS NOT NULL THEN
                    id:=r.qabdbrid;
                    parent:=r.spdbrid;
                    identname:='qabService.q_nr';
                    ident:=r.q_nr;
                    subject:=NULL;
                    dat:=NULL;
                    done:=NULL;
                    RETURN NEXT;
                    INSERT INTO tmpdata(tname, tdbrid) VALUES ('qab', r.qabdbrid);
            END IF;
     END LOOP;
     --Arbeitspakete und Arbeitsgänge dazu
     FOR r IN SELECT tname, tdbrid, ab_ix FROM tmpdata JOIN abk ON ab_tablename=tname AND ab_dbrid=tdbrid WHERE tdbrid IS NOT NULL AND ab_parentabk IS NULL ORDER BY pos LOOP
            FOR r1 IN SELECT ab_ix, ab_ap_nr, ab_parentabk, ab_done FROM abk WHERE
                                    ab_ix IN (SELECT * FROM tplanterm.get_all_child_abk(r.ab_ix))
                                    ORDER BY ab_pos LOOP
                    ismyrec :=FALSE;
                    id      :=r1.ab_ix;
                    parent  :=IFTHEN(r1.ab_parentabk IS NULL, r.tdbrid, r1.ab_parentabk::VARCHAR);
                    identname:='abk.ab_ix';
                    ident   :=r1.ab_ix;
                    subject :=r1.ab_ap_nr;
                    dat     :=NULL;
                    txt     :=NULL;
                    done    :=r1.ab_done;
                    RETURN NEXT;
                    --Arbeitsgänge
                    FOR r2 IN SELECT dbrid, a2_id, a2_n, a2_ende, a2_subject, a2_txt FROM ab2 WHERE a2_ab_ix=r1.ab_ix ORDER BY a2_n LOOP
                            ismyrec :=r2.a2_id=a2id;
                            id      :=r2.dbrid;
                            parent  :=r1.ab_ix;
                            identname:='ab2';
                            ident   :=r2.a2_n;
                            subject :=r2.a2_subject;
                            dat     :=NULL;
                            txt     :=r2.a2_txt;
                            done    :=r2.a2_ende;
                            RETURN NEXT;
                            --Rückgabeparameter
                            FOR r4 IN SELECT dbrid, a2r_vname, a2r_vbez, a2r_value, a2r_bem FROM ab2_resultparam WHERE a2r_a2_id=r2.a2_id ORDER BY a2r_vbez DESC LOOP
                                    ismyrec :=FALSE;
                                    id      :=r4.dbrid;
                                    parent  :=r2.dbrid;
                                    identname:='ab2_resultparam';
                                    ident   :=r4.a2r_vbez;
                                    subject :=r4.a2r_value;
                                    dat     :=NULL;
                                    txt     :=r4.a2r_bem;
                                    done    :=NULL;
                                    RETURN NEXT;
                            END LOOP;
                            --Stempelungen
                            FOR r3 IN SELECT bdea.dbrid, nameAufloesen(ba_minr::VARCHAR), ba_ende, ba_anf, o2k_bez, ba_txt FROM bdea LEFT JOIN op2kat ON o2k_id=ba_o2k_id WHERE ba_ix=r1.ab_ix AND ba_op=r2.a2_n ORDER BY ba_anf DESC LOOP
                                    ismyrec :=FALSE;
                                    id      :=r3.dbrid;
                                    parent  :=r2.dbrid;
                                    identname:='bdea';
                                    ident   :=r3.nameAufloesen;
                                    subject :=r3.o2k_bez;
                                    dat     :=r3.ba_anf;
                                    txt     :=r3.ba_txt;
                                    done    :=r3.ba_ende;
                                    RETURN NEXT;
                            END LOOP;
                    END LOOP;
            END LOOP;
     END LOOP;
     --
     DROP TABLE tmpdata;
     --
     RETURN;
    END $$ LANGUAGE plpgsql;
--

-- Daten rund um den Fertigungsartikel zusammensuchen anhand einer Arbeitsgang-ID
  -- DROP FUNCTION IF EXISTS tabk.abk__fertdata__by__a2_id(INTEGER);
CREATE OR REPLACE FUNCTION tabk.abk__fertdata__by__a2_id(
    IN  a2id integer,
    OUT a2_id integer,          OUT op_ix integer,          OUT o2_id integer,         OUT a2_ab_ix integer,        OUT a2_n integer,
    OUT apaknr    varchar(40),  OUT apakbez   varchar(100),
    OUT fertaknr  varchar(40),  OUT fertakbez varchar(100),  OUT fertakznr varchar(50),  OUT fertakidx varchar(30),  OUT fertakdim varchar(50), OUT fertakmat varchar(50),
    OUT fertauftg varchar(30),  OUT fertauftgpos integer,    OUT fertakdin varchar(50)
    )
    RETURNS record
    AS $$
    DECLARE rec record;
    BEGIN
      -- Laufzeit so : 200ms,
      -- Mit CASE WHEN a2_o2_id IS NULL THEN > 30 Sekunden

      SELECT
        ab2.a2_id, opl.op_ix, op2.o2_id, ab2.a2_ab_ix, ab2.a2_n,
        a2_aknr, COALESCE(a2_awtx,apart.ak_bez), fart.ak_nr, fart.ak_bez,
        fart.ak_znr, COALESCE(ld_aknr_idx, fart.ak_idx), fart.ak_dim, fart.ak_mat, ld_auftg, ld_pos, fart.ak_din
      INTO
        a2_id, op_ix, o2_id,a2_ab_ix,a2_n,
        apaknr, apakbez, fertaknr, fertakbez,
        fertakznr, fertakidx, fertakdim, fertakmat,
        fertauftg, fertauftgpos, fertakdin
      FROM public.ab2
        JOIN public.abk ON ab2.a2_ab_ix = abk.ab_ix
        LEFT JOIN public.ldsdok ON ld_code = 'I' AND ld_abk = abk.ab_ix
        LEFT JOIN public.opl ON opl.op_ix = abk.ab_askix
        LEFT JOIN public.art AS fart  ON fart.ak_nr = COALESCE(op_n,ab_ap_nr, ld_aknr) -- Fertigungsartikel: Nicht immer ASK vorhanden (Nacharbeit etc./ ab_ap_nr enthält dann Artikelnummer
        LEFT JOIN public.art AS apart ON apart.ak_nr = a2_aknr                         -- Arbeitspaket
        LEFT JOIN public.op2 ON op2.o2_id = ab2.a2_o2_id
      WHERE ab2.a2_id = a2id
      ORDER BY ab2.a2_ab_ix, FertAknr;
    END $$ LANGUAGE plpgsql STABLE STRICT;
--

-- Berechnung des spezifischen Material-Bedarfstermins (ag_kdatum) der Materialposition.
--  Bei:
--       Zuweisen der AG Nr zur Materiallistenposition
--       agmi_dat, agmi_vtermweek, agmi_beistell

CREATE OR REPLACE FUNCTION tabk.abk__auftgi_kldatum__calc(
    IN  _ag_ids         integer[],
    OUT agid            integer,
    OUT kdatum_new      date,
    OUT kdatuml_bedarf  date
  ) RETURNS SETOF record
  AS $$
  DECLARE mat_pos record;
  BEGIN
    IF coalesce(cardinality(_ag_ids), 0) < 1 THEN   RETURN; END IF; -- ohne Ziel raus

    FOR mat_pos IN
        SELECT
          ag_id,                                    -- MatPos
          ac_auftgi_pre_float,                      -- Pufferzeit laut AC
          agmi_v_id, agmi_dat, agmi_vtermweek,      -- MatPos-Zusatzinfos
          ab_ix, ab_stat, ab_at,                    -- übergeordnete ABK
          a2_at,                                    -- verlinkter ABK-AG
          ld_code, timediff_substdays(coalesce(ld_terml, ld_term), ak_bfr + ac_auftgi_pre_float) AS ldsdok_beistellung_term-- übergeordnete Beistellbestellung. Beschaffungsfrist dieser Bestellung selbst. Beschaffungsfrist der Beistellung wird in Bedarfsberechnung!
        FROM auftg
          JOIN art ON ak_nr = ag_aknr
          JOIN artcod ON ac_n = ak_ac
          JOIN abk ON ab_ix = ag_parentabk
          LEFT JOIN ldsdok ON ld_abk = ab_ix
          LEFT JOIN auftgmatinfo ON agmi_ag_id = ag_id
          LEFT JOIN ab2 ON a2_id = ag_a2_id
        WHERE ag_astat = 'I'
          AND ag_id = ANY(_ag_ids)  -- für alle angg. MatPos
        ORDER BY ag_id
    LOOP
        agid := mat_pos.ag_id;
        kdatum_new := NULL;
        kdatuml_bedarf := NULL;

        -- Neuen Bedarfstermin der MatPos ermitteln.
            IF (mat_pos.agmi_dat IS NOT NULL OR mat_pos.agmi_vtermweek IS NOT NULL) THEN  -- Führender Termin ist direkt in MatPos-Zusatzinfos angg..
                kdatum_new := coalesce(mat_pos.agmi_dat, termweek_to_date(mat_pos.agmi_vtermweek));
                kdatuml_bedarf := kdatum_new;

                IF NOT EXISTS(SELECT true FROM versart WHERE v_id = mat_pos.agmi_v_id AND v_stat = 'L') THEN  -- Wenn kein Versand durch Lieferanten,
                    kdatum_new :=                                                                             -- dann n Werktage Pufferzeit vorziehen. Mit Versand durch Lieferanten 0.
                      timediff_substdays(
                        kdatum_new,
                        mat_pos.ac_auftgi_pre_float, true, true
                    );
                END IF;

            -- Anhand ABK-Material-Termin bzw. Terminverschiebung per Plantafel-Drag&Drop (tplanterm.move_ab2).
            -- Es ist kein führender Termin in MatPos-Zusatzinfos angg..
            ELSE

                -- Termin aus AG der ABK
                kdatuml_bedarf := coalesce(
                                      -- Termin laut ABK-AG. Nur angejoint wenn ag_a2_id
                                      mat_pos.a2_at::date,
                                      -- erster planrelevanter AG. Siehe abk__at_et__from__ab2__sync
                                      mat_pos.ab_at
                                  );

                 -- BEISTELLUNG und kein AG => aus übergeordneter Beistellbestellung nehmen!
                IF kdatuml_bedarf IS null AND mat_pos.ld_code = 'E' THEN
                    kdatuml_bedarf := mat_pos.ldsdok_beistellung_term;
                END IF;

                kdatuml_bedarf := timediff_substdays(kdatuml_bedarf, 5, true, true); -- eigentliches Bedarfsdatum 5 Tage bevor es auf Maschine / zur Auslieferung (Beistellung) muss!

                kdatum_new :=
                    -- Termin inkl. n Werktage Pufferzeit ermitteln
                    timediff_substdays(
                        kdatuml_bedarf,

                        -- n Werktage Pufferzeit vorziehen
                        mat_pos.ac_auftgi_pre_float, true, true
                    )
                ;


                IF NOT (   TSystem.ENUM_GetValue(mat_pos.ab_stat, 'PS')
                        OR EXISTS(SELECT true FROM ldsdok WHERE ld_abk = mat_pos.ab_ix)
                        )
                THEN                  -- Wenn kein Produktionsauftrag sowie keine Set-ABK,
                    kdatum_new := timediff_substdays(kdatum_new,  TSystem.Settings__GetInteger('ABK.AuftgI.Projekt.Pufferzeit.global', 10), true, true);    -- dann ist es eine Projekt-ABK und damit weitere 14 Tage vorziehen, sonst nur Pufferzeit lauf AC.
                END IF;
            END IF;
        --

        RETURN NEXT;
    END LOOP;
  END $$ LANGUAGE plpgsql STABLE;

CREATE OR REPLACE FUNCTION tabk.abk__auftgi_kdatum__calc(
    IN  _ag_ids         integer[],
    OUT agid            integer,
    OUT kdatum_new      date,
    OUT kdatuml_bedarf  date
  ) RETURNS SETOF record
  AS $$
     SELECT * FROM tabk.abk__auftgi_kldatum__calc(_ag_ids);
  $$ LANGUAGE sql;
--

-- Bedarfstermine aktualisieren für bestimmte ABK-Materiallisten-Positionen.
CREATE OR REPLACE FUNCTION tabk.abk__auftgi_kldatum__refresh(_ag_ids integer[]) RETURNS void AS $$
  BEGIN
    IF COALESCE(cardinality(_ag_ids), 0) < 1 THEN   RETURN; END IF; -- ohne Ziel raus

    UPDATE auftg
       SET ag_kdatum = kdatum_new,     -- inkl. Puffer
           ag_ldatum = kdatuml_bedarf  -- Tatsächlicher Bedarfstermin zur Fertigung
      FROM tabk.abk__auftgi_kldatum__calc(_ag_ids)
     WHERE ag_id = agid
       AND NOT ag_done
       AND (   ag_kdatum IS DISTINCT FROM kdatum_new
            OR ag_ldatum IS DISTINCT FROM kdatuml_bedarf
            );

    RETURN;
  END $$ LANGUAGE plpgsql VOLATILE;

CREATE OR REPLACE FUNCTION tabk.abk__auftgi_kdatum__refresh(
    IN  _ag_ids         integer[]
  ) RETURNS void
  AS $$
     SELECT * FROM tabk.abk__auftgi_kldatum__refresh(_ag_ids);
  $$ LANGUAGE sql;
--

-- Bedarfstermine aktualisieren für bestimmte ABK-Materialliste einer ABK.
CREATE OR REPLACE FUNCTION tabk.abk__auftgi_kldatum__refresh_by__ag_parent_abk(_parent_abk integer) RETURNS void AS $$
  BEGIN
    IF _parent_abk IS NULL THEN   RETURN; END IF;

    PERFORM tabk.abk__auftgi_kldatum__refresh(ag_ids)
    FROM (
        SELECT array_agg(ag_id) AS ag_ids
          FROM auftg
         WHERE ag_astat = 'I'
           AND ag_parentabk = _parent_abk
     ) AS sub
    ;

    RETURN;
  END $$ LANGUAGE plpgsql VOLATILE;

CREATE OR REPLACE FUNCTION tabk.abk__auftgi_kdatum__refresh_by__ag_parent_abk(_parent_abk integer) RETURNS void
  AS $$
     SELECT * FROM tabk.abk__auftgi_kldatum__refresh_by__ag_parent_abk(_parent_abk);
  $$ LANGUAGE sql;
--



-- Ermittelt für eine Material-Position (aus PA oder Fertigungsprojekt) die zugehörigen offenen BANF-Positionen über den Kundenauftrag (Angebot oder Auftrag).
CREATE OR REPLACE FUNCTION tabk.abk__auftgi__bestanfpos__get(in_ag_id INTEGER) RETURNS SETOF INTEGER AS $$
  DECLARE mat_pos RECORD;
  BEGIN
    -- Material-Position
    SELECT ag_parentabk, ag_aknr
    INTO mat_pos
    FROM auftg
    WHERE ag_id = in_ag_id
      AND ag_astat = 'I'
    ;

    IF mat_pos IS NULL THEN   RETURN;   END IF; -- Keine valide Quelle, dann raus.

    RETURN QUERY
      SELECT bap_id
      FROM auftg -- Kundenauftrag
        JOIN bestanfpos ON bap_agnr = ag_nr AND bap_agpos = ag_pos -- BANF-Pos. wurde über Auftragspos. erstellt.
      WHERE ag_id = tplanterm.abk_main_auftg_id(mat_pos.ag_parentabk) -- Kundenauftrag aus ABK ermitteln.
        AND ag_astat IN ('A', 'E')      -- Angebot oder Auftrag
        AND bap_aknr = mat_pos.ag_aknr  -- für den gleichen Artikel
        AND NOT bap_done                -- nur offene BANF-Pos.
      ORDER BY bap_id
    ;
  END $$ LANGUAGE plpgsql STABLE STRICT;
--

-- Ermittelt für eine BANF-Position die zugehörigen Material-Positionen (aus PA oder Fertigungsprojekt) über den Kundenauftrag (Angebot oder Auftrag).
CREATE OR REPLACE FUNCTION tabk.bestanfpos__abk__auftgi__get(in_bap_id INTEGER) RETURNS SETOF INTEGER AS $$
  DECLARE bap_pos RECORD;
  BEGIN
    -- BANF-Position und zug. Kundenauftrag (KA)
    SELECT ag_id AS ka_ag_id, bap_aknr
    INTO bap_pos
    FROM bestanfpos
      JOIN auftg ON ag_nr = bap_agnr AND ag_pos = bap_agpos -- BANF-Pos. wurde über Auftragspos. erstellt.
    WHERE bap_id = in_bap_id
      AND ag_astat IN ('A', 'E') -- Angebot oder Auftrag
    ;

    IF bap_pos IS NULL THEN   RETURN;   END IF; -- Keine valide Quelle, dann raus.

    RETURN QUERY
      SELECT ag_id
      FROM auftg -- Material-Position
      WHERE ag_parentabk IN (SELECT tplanterm.get_all_child_abk(tplanterm.auftg_get_abk(bap_pos.ka_ag_id))) -- komplette ABK-Struktur aus Haupt-ABK vom Kundenauftrag ermitteln.
        AND ag_astat = 'I'
        AND ag_aknr = bap_pos.bap_aknr  -- für den gleichen Artikel
      ORDER BY ag_id
    ;
  END $$ LANGUAGE plpgsql STABLE STRICT;
--

--
CREATE OR REPLACE FUNCTION tabk.abk__auftgi__ag_a2_id__by__stv_op2__create(
    IN _ab_ix            integer,
    IN _force_new_links  boolean = false,
       -- alle Materiallistenpositionen ohne Zuordnung erhalten automatisch den ersten planrelevanten AG. Könnte eigentlich standard Verhalten werden? Relevant für Bestellvorschlag Standortbezogen
    IN _force_first_planrelevant_ag boolean = TSystem.Settings__GetBool('auftgi__ag_a2_id__first_planrelevant__auto')
    )
    RETURNS void
    AS $$
    DECLARE
       _record   record;
       _a2_id_planrelevant_first integer;
    BEGIN
        -- Erstellt und aktualisiert die Verknüpfung der Materialpositionen (auftgi) mit den Arbeitsgängen (ab2) einer ABK.
        -- Gemäß der AVOR-Vorgaben (stv_op2) nach Stücklistenauflösung und ABK-Erstellung.


        -- Ohne Angabe von ABK Fehler 16850
        IF _ab_ix IS NULL THEN
            RAISE EXCEPTION '%', lang_text( 16850 );
        END IF;


        -- Komplette Verlinkung neu aufbauen:
          -- Wenn Informationen der aktuellen Auflösung zur zug. Verlinkung vorhanden sind.
          -- Damit bestehende Verlinkungen nicht gelöscht werden, wenn überhaupt keine Auflösungs-Infos vorhanden sind.
        IF _force_new_links THEN

            -- aktuelle Stücklistenauflösung mit Verlinkung vorhanden
            IF  EXISTS(
                    SELECT true
                      FROM auftg
                      JOIN stvtrs   ON stvtrs.auftg_ag_id = auftg.ag_id
                      JOIN stv      ON stv.dbrid = stvtrs.stv_dbrid
                     WHERE ag_parentabk = _ab_ix
                )
            THEN

                -- ALLE bestehenden Verlinkungen entfernen
                UPDATE auftg
                   SET ag_a2_id = null
                 WHERE ag_parentabk = _ab_ix
                   AND ag_a2_id IS NOT NULL
                ;

            END IF;

        END IF;


        -- Verlinkungen aufbauen
        FOR _record IN

            -- Verlinkungen ermitteln
            SELECT ag_id,
                   a2_id

              FROM -- ABK-Materialpositionen
                   auftg
                   -- aus Stücklistenauflösung (zur ABK-Erstellung verwendet)
              JOIN stvtrs   ON stvtrs.auftg_ag_id = auftg.ag_id
                   -- der zug. Stücklisten-Position
              JOIN stv      ON stv.dbrid = stvtrs.stv_dbrid
                   -- mit Info zur Verlinkung von Stücklisten-Position und ASK-AG
              JOIN stv_op2  ON stv_op2.sto2_st_id = stv.st_id
                   -- der zug. ABK-AG des ASK-AG
              JOIN ab2      ON ab2.a2_ab_ix = auftg.ag_parentabk AND ab2.a2_o2_id = stv_op2.sto2_o2_id

             WHERE -- der gesamten ABK
                   ag_parentabk = _ab_ix
             ORDER BY ag_pos, ag_id

        LOOP

            -- Verlinkungen entspr. Ermittlung setzen
            UPDATE auftg
               SET ag_a2_id = _record.a2_id
             WHERE ag_id    = _record.ag_id
               AND ag_a2_id IS NULL
            ;

        END LOOP;

        IF _force_first_planrelevant_ag  THEN
            -- ersten AG welcher planrelevant ist ermitteln
            _a2_id_planrelevant_first := tabk.abk__ab2__a2_id__ks_plan__by__ab_ix(_ab_ix);
            -- alles Material ohne Verlinkung dem ersten planrelevanten AG zuordnen
            IF _a2_id_planrelevant_first IS NOT null THEN
                FOR _record IN
                    SELECT ag_id
                      FROM auftg
                     WHERE ag_parentabk = _ab_ix
                       AND ag_a2_id IS null
                LOOP
                    UPDATE auftg
                       SET ag_a2_id = _a2_id_planrelevant_first
                     WHERE ag_a2_id IS null
                       AND ag_parentabk = _ab_ix
                       AND NOT ag_done;
                END LOOP;
            END IF;
        END IF;

        RETURN;
    END $$ LANGUAGE plpgsql VOLATILE;
--

--- #18577,22103 Hilfsfunktion, suche aktuelle(n) ABK/AG pro SN
CREATE OR REPLACE FUNCTION tabk.mapsernr__ab_ix__a2_n__get_actual(
    IN  _ab_ix        integer,
    IN  _Seriennummer varchar,
    OUT ab_ix         integer,
    OUT a2_n          integer  -- Arbeitsgang der als nächstes für die Seriennummer ansteht
  ) RETURNS RECORD
  AS $$

    WITH abks_ags AS (
      -- Hole alle aktiven ABKs und Arbeitsgänge für die Seriennummer
      --- Vorgabeseriennummernnummer aus der Mapsernr-Tabelle
      SELECT ab_ix,
             a2_n,
             a2_id,
             lgs_sernr,
             lgs_id,
             TLager.lagsernr__sn_ab2_mapping__by__a2_id__get(a2_id) AS sn_specific_ab2
        FROM mapsernr
        JOIN lagsernr ON ms_lgs_id = lgs_id AND ms_table = 'ldsdok'::REGCLASS   -- Vorgabe-SN Verlinkung
        JOIN ldsdok ON ms_pkey::int = ld_id
        JOIN abk ON ab_ld_id = ld_id AND NOT ab_done                            -- Nur aktive ABKs
        JOIN ab2 ON a2_ab_ix = abk.ab_ix AND NOT a2_ende                        -- Nur aktive Arbeitsgänge
       WHERE lgs_sernr ILIKE _Seriennummer
    ),
    completed_ops AS (
      -- Hole abgeschlossene Vorgänge für diese Seriennummer
      SELECT DISTINCT ba_ix, ba_op
        FROM bdea
        JOIN abks_ags ON ba_op = a2_n AND ba_ix = ab_ix                         -- nur Rückmeldungen zu den Arbeitsgängen der SN
        JOIN mapsernr ON ms_pkey = ba_id AND ms_table = 'bdea'::REGCLASS
       WHERE ms_lgs_id = lgs_id                                                 -- nur Rückmeldungen zu der SN
    )

    SELECT ab_ix
         , a2_n
      FROM abks_ags
      LEFT JOIN completed_ops ON abks_ags.ab_ix = completed_ops.ba_ix
                             AND abks_ags.a2_n = completed_ops.ba_op

     WHERE completed_ops.ba_ix IS NULL                                    -- nur Arbeitsgänge zu denen es noch keine Rückmeldung gab
       AND ( lgs_id = ANY( sn_specific_ab2 ) OR sn_specific_ab2 IS NULL ) -- nur Arbeitsgänge die auch für die Seriennummer relevant sind

     ORDER BY ab_ix = _ab_ix DESC,  -- Priorisiere Übereinstimmungen mit Input ABK
              ab_ix,                -- Dann sortiere nach ABK Index
              a2_n                  -- Zuletzt nach Arbeitsgang-Nummer
     LIMIT 1;

  $$ LANGUAGE sql STABLE PARALLEL SAFE;
---

--- #21278 Assistent Projekt: Arbeitspakete anstatt Kostenstellen
CREATE OR REPLACE FUNCTION tabk.abk__ab_ap_nr__child__offen(
    IN _ab_ix integer
    )
    RETURNS varchar(500)
    AS $$
       SELECT string_agg( ab_ap_nr, '->' )
       FROM ( SELECT ab_ap_nr
              FROM abk
              WHERE ab_tablename = 'anl'
                AND ab_mainabk = _ab_ix
                AND not coalesce( ab_done, false )
              ORDER BY ab_pos NULLS LAST, abk.ab_ix
            ) AS sub;
    $$ LANGUAGE sql STABLE STRICT PARALLEL SAFE;

--
CREATE OR REPLACE FUNCTION tabk.tableident_alllinkfields(
    IN  _tablename      varchar,
    IN  _dbrid          varchar,
    -- Projekt
    OUT an_nr           varchar,
    -- Auftrag
    OUT ag_astat        varchar,
    OUT ag_nr           varchar,
    OUT ag_pos          integer,
    -- Einkauf
    OUT ld_code         varchar,
    OUT ld_auftg        varchar,
    OUT ld_pos          integer,
    -- Kundenanfrage
    OUT kanf_nr         varchar
  )
  RETURNS record
  AS $$

    SELECT an_nr
         , at_astat AS ag_astat
         , at_nr AS ag_nr
         , ( SELECT min( ag_pos ) FROM auftg WHERE at_nr = ag_nr AND at_astat = ag_astat ) AS ag_pos
         , ld_code
         , ld_auftg
         , ld_pos
         , kanf_nr
      FROM abk
      LEFT JOIN anl         ON _tablename = 'anl'         AND anl.dbrid         = _dbrid -- Projekt
      LEFT JOIN auftgtxt    ON _tablename = 'auftgtxt'    AND auftgtxt.dbrid    = _dbrid -- Auftrag
      LEFT JOIN ldsdok      ON _tablename = 'ldsdok'      AND ldsdok.dbrid      = _dbrid -- Einkauf
      LEFT JOIN kundanfrage ON _tablename = 'kundanfrage' AND kundanfrage.dbrid = _dbrid -- Interessentenanfrage

  $$ LANGUAGE sql STABLE STRICT PARALLEL SAFE;

COMMENT ON FUNCTION tabk.tableident_alllinkfields(character varying, character varying)
  IS 'Gibt zu einem Tabellennamen (_tablename) und der dbrid (_dbrid) die für die Verlinkung relevanten Felder aus Projekt, Auftrag, Bestellung und Interessantenanfrage zurück.';
--

--- #23176
CREATE OR REPLACE FUNCTION tabk.ksv__ks_abt__change(
    IN _ks_abt_old   varchar,       ---
    IN _ks_abt_new   varchar
    )
    RETURNS void AS $$
    BEGIN

        PERFORM TSystem.triggers__foreign_key__disable__user('ksv');

        UPDATE ksv SET ks_abt = _ks_abt_new WHERE ks_abt = _ks_abt_old; -- ON UPDATE CASCADE!

        PERFORM TSystem.trigger__disable('ab2_wkstplan', 'USER'); -- secpo-check / copy exactly
        UPDATE ab2_wkstplan SET a2w_oks = _ks_abt_new, a2w_ks = REPLACE(a2w_ks, _ks_abt_old, _ks_abt_new) WHERE a2w_oks = _ks_abt_old;
        PERFORM TSystem.trigger__enable('ab2_wkstplan', 'USER');

        PERFORM TSystem.triggers__foreign_key__enable__user('ksv');

        RETURN;

    END $$ LANGUAGE plpgsql;
